import * as ts from 'typescript';

import ITranslations from '../../types/ITranslations';
import ITranslationKeySet from '../../types/ITranslationKeySet';

import getFunctionName from '../../utilities/getFunctionName';

/**
 * Собирает файл с функцией i18n которая соответствует старому интерфейсу из ymaps-tanker
 * (keySetName: string, keyName: string, params?: object)
 */
export default class OldFunctionBuilder {
    private readonly factory: ts.NodeFactory = ts.factory;

    build(translations: ITranslations): string {
        const resultFile = ts.createSourceFile(
            '',
            '',
            ts.ScriptTarget.Latest,
            false,
            ts.ScriptKind.TS,
        );
        const printer = ts.createPrinter({newLine: ts.NewLineKind.LineFeed});

        const ast = this.buildAST(translations);

        return printer.printList(ts.ListFormat.MultiLine, ast, resultFile);
    }

    private buildAST(
        translations: ITranslations,
    ): ts.NodeArray<
        ts.ImportDeclaration | ts.VariableStatement | ts.FunctionDeclaration
    > {
        const {factory} = this;

        const keySets = Object.values(translations.keySets).filter(
            keySet => Object.keys(keySet.keys).length > 0,
        );

        return factory.createNodeArray([
            ...this.buildASTImports(keySets),
            this.buildASTMapObject(keySets),
            this.buildASTFunction(),
        ]);
    }

    private buildASTImports(
        keySets: ITranslationKeySet[],
    ): ts.ImportDeclaration[] {
        const {factory} = this;

        return keySets.map(keySet =>
            factory.createImportDeclaration(
                undefined,
                undefined,
                factory.createImportClause(
                    false,
                    undefined,
                    factory.createNamespaceImport(
                        factory.createIdentifier(getFunctionName(keySet.name)),
                    ),
                ),
                factory.createStringLiteral(`./${keySet.name}`),
            ),
        );
    }

    private buildASTMapObject(
        keySets: ITranslationKeySet[],
    ): ts.VariableStatement {
        const {factory} = this;

        return factory.createVariableStatement(
            undefined,
            factory.createVariableDeclarationList(
                [
                    factory.createVariableDeclaration(
                        factory.createIdentifier('mapKeySetNameToKeySet'),
                        undefined,
                        undefined,
                        factory.createObjectLiteralExpression(
                            keySets.map(keySet =>
                                factory.createPropertyAssignment(
                                    factory.createComputedPropertyName(
                                        factory.createStringLiteral(
                                            keySet.name,
                                        ),
                                    ),
                                    factory.createIdentifier(
                                        getFunctionName(keySet.name),
                                    ),
                                ),
                            ),
                            true,
                        ),
                    ),
                ],
                ts.NodeFlags.Const,
            ),
        );
    }

    private buildASTFunction(): ts.FunctionDeclaration {
        const {factory} = this;

        return factory.createFunctionDeclaration(
            undefined,
            [
                factory.createModifier(ts.SyntaxKind.ExportKeyword),
                factory.createModifier(ts.SyntaxKind.DefaultKeyword),
            ],
            undefined,
            factory.createIdentifier('i18n'),
            undefined,
            [
                factory.createParameterDeclaration(
                    undefined,
                    undefined,
                    undefined,
                    factory.createIdentifier('keySetName'),
                    undefined,
                    factory.createTypeOperatorNode(
                        ts.SyntaxKind.KeyOfKeyword,
                        factory.createTypeQueryNode(
                            factory.createIdentifier('mapKeySetNameToKeySet'),
                        ),
                    ),
                    undefined,
                ),
                factory.createParameterDeclaration(
                    undefined,
                    undefined,
                    undefined,
                    factory.createIdentifier('keyFunctionName'),
                    undefined,
                    factory.createKeywordTypeNode(ts.SyntaxKind.StringKeyword),
                    undefined,
                ),
                factory.createParameterDeclaration(
                    undefined,
                    undefined,
                    undefined,
                    factory.createIdentifier('params'),
                    undefined,
                    factory.createKeywordTypeNode(ts.SyntaxKind.ObjectKeyword),
                    factory.createObjectLiteralExpression([], false),
                ),
            ],
            undefined,
            factory.createBlock(
                [
                    factory.createVariableStatement(
                        undefined,
                        factory.createVariableDeclarationList(
                            [
                                factory.createVariableDeclaration(
                                    factory.createIdentifier('keySet'),
                                    undefined,
                                    factory.createKeywordTypeNode(
                                        ts.SyntaxKind.AnyKeyword,
                                    ),
                                    factory.createElementAccessExpression(
                                        factory.createIdentifier(
                                            'mapKeySetNameToKeySet',
                                        ),
                                        factory.createIdentifier('keySetName'),
                                    ),
                                ),
                            ],
                            ts.NodeFlags.Const,
                        ),
                    ),
                    factory.createIfStatement(
                        factory.createPrefixUnaryExpression(
                            ts.SyntaxKind.ExclamationToken,
                            factory.createIdentifier('keySet'),
                        ),
                        factory.createBlock(
                            [
                                factory.createThrowStatement(
                                    factory.createNewExpression(
                                        factory.createIdentifier('Error'),
                                        undefined,
                                        [
                                            factory.createTemplateExpression(
                                                factory.createTemplateHead(
                                                    'KeySet ',
                                                    'KeySet ',
                                                ),
                                                [
                                                    factory.createTemplateSpan(
                                                        factory.createIdentifier(
                                                            'keySetName',
                                                        ),
                                                        factory.createTemplateTail(
                                                            ' not found',
                                                            ' not found',
                                                        ),
                                                    ),
                                                ],
                                            ),
                                        ],
                                    ),
                                ),
                            ],
                            true,
                        ),
                        undefined,
                    ),
                    factory.createVariableStatement(
                        undefined,
                        factory.createVariableDeclarationList(
                            [
                                factory.createVariableDeclaration(
                                    factory.createIdentifier('keyFunction'),
                                    undefined,
                                    undefined,
                                    factory.createElementAccessExpression(
                                        factory.createIdentifier('keySet'),
                                        factory.createIdentifier(
                                            'keyFunctionName',
                                        ),
                                    ),
                                ),
                            ],
                            ts.NodeFlags.Const,
                        ),
                    ),
                    factory.createIfStatement(
                        factory.createPrefixUnaryExpression(
                            ts.SyntaxKind.ExclamationToken,
                            factory.createIdentifier('keyFunction'),
                        ),
                        factory.createBlock(
                            [
                                factory.createThrowStatement(
                                    factory.createNewExpression(
                                        factory.createIdentifier('Error'),
                                        undefined,
                                        [
                                            factory.createTemplateExpression(
                                                factory.createTemplateHead(
                                                    'Key ',
                                                    'Key ',
                                                ),
                                                [
                                                    factory.createTemplateSpan(
                                                        factory.createIdentifier(
                                                            'keyFunctionName',
                                                        ),
                                                        factory.createTemplateMiddle(
                                                            ' not found in keySet ',
                                                            ' not found in keySet ',
                                                        ),
                                                    ),
                                                    factory.createTemplateSpan(
                                                        factory.createIdentifier(
                                                            'keySetName',
                                                        ),
                                                        factory.createTemplateTail(
                                                            '',
                                                            '',
                                                        ),
                                                    ),
                                                ],
                                            ),
                                        ],
                                    ),
                                ),
                            ],
                            true,
                        ),
                        undefined,
                    ),
                    factory.createReturnStatement(
                        factory.createCallExpression(
                            factory.createIdentifier('keyFunction'),
                            undefined,
                            [factory.createIdentifier('params')],
                        ),
                    ),
                ],
                true,
            ),
        );
    }
}
