import ITankerLinkOptions from '../../types/ITankerLinkOptions';

interface ILinkQuery {
    project: string;
    branch: string;
    keyset?: string;
    key?: string;
}

export default class TankerLinkBuilder {
    private readonly base: string;
    private readonly project: string;
    private readonly branch: string;
    private readonly isNewTanker: boolean;

    constructor({
        base = 'https://tanker.yandex-team.ru',
        branch = 'master',
        project,
        isNewTanker = false,
    }: ITankerLinkOptions) {
        this.base = base;
        this.project = project;
        this.branch = branch;
        this.isNewTanker = isNewTanker;
    }

    build(keySetName: string, keyName?: string): string {
        return this.isNewTanker
            ? this.buildNew(keySetName, keyName)
            : this.buildOld(keySetName, keyName);
    }

    /**
     * Формат старого танкера
     */
    private buildOld(keySetName: string, keyName?: string): string {
        const link = `${this.base}/?`;

        const query: ILinkQuery = {
            project: this.project,
            branch: this.branch,
            keyset: keySetName,
            key: keyName,
        };

        return `${link}${Object.entries(query)
            .filter(([, value]) => value)
            .map(([key, value]) => `${key}=${value}`)
            .join('&')}`;
    }

    /**
     * Собрать в формате для нового танкера
     */
    private buildNew(keySetName: string, keyName?: string): string {
        let link = `${this.base}/project/${this.project}/keyset/${keySetName}`;

        if (keyName) {
            link += `/key/${keyName}`;
        }

        return `${link}?branch=${this.branch}`;
    }
}
