import axios, {AxiosInstance} from 'axios';

import ITankerOptions from '../../types/ITankerOptions';
import ITranslationsResponse from './types/ITranslationsResponse';
import ELanguages from '../../types/ELanguages';
import ITranslationsRequest from './types/ITranslationsRequest';
import ITranslations from '../../types/ITranslations';
import ITranslationKeySet from '../../types/ITranslationKeySet';
import ITranslationKey from '../../types/ITranslationKey';
import ITranslationLangKey from '../../types/ITranslationLangKey';
import IPrepareTranslationFunction from '../../types/IPrepareTranslationFunction';

interface ITankerDownloaderOptions extends ITankerOptions {
    /**
     * Ссылка до api танкера
     * @default https://tanker-api.yandex-team.ru
     */
    baseUrl?: string;

    /**
     * Токен для доступа в танкер
     */
    token?: string;

    /**
     * Cтатус переводов
     */
    status?: 'unapproved';

    /**
     * Языки переводов для скачивания
     * Если не передан, то будут скачаны все
     */
    languages?: ELanguages[];

    /**
     * Функция которая подготавливает перевод из танкера
     */
    prepareTranslation?: IPrepareTranslationFunction;
}

/**
 * Загрузка переводов из танкера
 * Использует старое API
 * @deprecated для загрузки переводов используйте пакет @yandex-data-ui/i18n-sync
 */
export default class TankerDownloader {
    private readonly baseUrl: string;
    private readonly project: string;
    private readonly token: string | undefined;
    private readonly branch: string;
    private readonly status: 'unapproved' | undefined;
    private readonly languages: ELanguages[] | undefined;
    private readonly prepareTranslation: IPrepareTranslationFunction;

    private readonly axios: AxiosInstance;

    constructor({
        baseUrl = 'https://tanker-api.yandex-team.ru',
        token,
        project,
        branch,
        status,
        languages,
        prepareTranslation = ({text}): string => text,
    }: ITankerDownloaderOptions) {
        this.baseUrl = baseUrl;
        this.project = project;
        this.token = token;
        this.branch = branch;
        this.status = status;
        this.languages = languages;
        this.prepareTranslation = prepareTranslation;

        this.axios = this.createAxiosInstance();
    }

    async download(): Promise<ITranslations> {
        const response = await this.loadTranslations();

        return this.prepareData(response);
    }

    private prepareData(response: ITranslationsResponse): ITranslations {
        const keySets = Object.entries(response.keysets).reduce(
            (accKeySets, [keySetName, originalKeySet]) => {
                const keySet: ITranslationKeySet = {
                    name: keySetName,
                    keys: Object.entries(originalKeySet.keys).reduce(
                        (accKeys, [keyName, originalKey]) => {
                            const key: ITranslationKey = {
                                name: keyName,
                                translations: Object.entries(
                                    originalKey.translations,
                                ).reduce(
                                    (
                                        accTranslations,
                                        [lang, originalTranslation],
                                    ) => {
                                        const sourceText =
                                            originalTranslation.form;
                                        const text = this.prepareTranslation({
                                            keyName,
                                            keySetName,
                                            lang: lang as ELanguages,
                                            text: sourceText,
                                        });
                                        const translation: ITranslationLangKey =
                                            {
                                                lang: lang as ELanguages,
                                                text,
                                                sourceText,
                                            };

                                        return {
                                            ...accTranslations,
                                            [lang as ELanguages]: translation,
                                        };
                                    },
                                    {},
                                ),
                            };

                            return {
                                ...accKeys,
                                [keyName]: key,
                            };
                        },
                        {},
                    ),
                };

                return {
                    ...accKeySets,
                    [keySetName]: keySet,
                };
            },
            {},
        );

        return {keySets};
    }

    private async loadTranslations(): Promise<ITranslationsResponse> {
        const params: ITranslationsRequest = {
            ['project-id']: this.project,
            ['branch-id']: this.branch,
            status: this.status,
            language: this.languages ? this.languages.join(',') : undefined,
        };

        const res = await this.axios.get<ITranslationsResponse>(
            'projects/export/tjson/',
            {
                params,
            },
        );

        return res.data;
    }

    private createAxiosInstance(): AxiosInstance {
        return axios.create({
            baseURL: this.baseUrl,
            headers: {
                Authorization: `OAuth ${this.token}`,
            },
        });
    }
}
