import fs from 'fs';

import {EProjectName} from 'server/constants/EProjectName';

import TRequest from 'server/types/TRequest';
import TResponse from 'server/types/TResponse';

import {
    FileNameError,
    OrderIdError,
    EmptyOrderIdError,
} from 'server/utilities/AppError/errors';
import fetchOrderAndWriteTicketsToDisk from './utilities/fetchOrderAndWriteTicketsToDisk';
import mergeAndWriteTicketToDisk from './utilities/mergeAndWriteTicketToDisk';

import TravelApi from '../../api/TravelApi/TravelApi';
import saveFileToS3 from '../../providers/s3Client/saveFileToS3';

export default async function generateBusesTickets(
    req: TRequest,
    res: TResponse,
): Promise<void> {
    const {
        body: {orderId, fileName},
        headers: {host},
        utils: logger,
        utils: {
            config: {s3Settings},
        },
    } = req;

    if (!host) {
        throw new Error('Host in headers is undefined');
    }

    if (typeof fileName !== 'string') {
        throw new FileNameError(fileName);
    }

    if (typeof orderId !== 'string') {
        throw new OrderIdError(orderId);
    }

    if (!orderId) {
        throw new EmptyOrderIdError(EProjectName.BUSES);
    }

    res.status(200).send();

    try {
        logger.log('BUSES_TICKETS_GENERATE_START', {orderId});

        // в названии файла есть /
        const encodedFileName = encodeURIComponent(fileName);

        const travelApi = new TravelApi(req);

        const ticketsPdfsPaths = await fetchOrderAndWriteTicketsToDisk({
            baseFileName: encodedFileName,
            orderId,
            travelApi,
            host,
            logger,
        });

        const mergedPdfTempLocation = await mergeAndWriteTicketToDisk(
            encodedFileName,
            ticketsPdfsPaths,
        );

        ticketsPdfsPaths.forEach(filesPath => {
            fs.unlinkSync(filesPath);
        });

        await saveFileToS3({
            fileContent: fs.readFileSync(mergedPdfTempLocation),
            fileName,
            orderId,
            s3Settings,
            logger,
        });

        fs.unlinkSync(mergedPdfTempLocation);

        logger.log('BUSES_TICKETS_GENERATE_SUCCESS', {orderId});
    } catch (err) {
        logger.logError('BUSES_TICKETS_GENERATE_FAILURE', {orderId, err});
    }
}
