import fs from 'fs';
import crypto from 'crypto';

import {getHotelsBusinessTripDocUrl} from 'constants/urls';

import TRequest from 'server/types/TRequest';
import TResponse from 'server/types/TResponse';

import AppError from 'server/utilities/AppError/AppError';
import {FileNameError, OrderIdError} from 'server/utilities/AppError/errors';

import puppeteerClient from 'server/providers/puppeteerClient/puppeteerClient';
import saveFileToS3 from 'server/providers/s3Client/saveFileToS3';

export default async function generateHotelBusinessTripDoc(
    req: TRequest,
    res: TResponse,
): Promise<void> {
    const {
        body: {orderId, fileName},
        headers: {host},
        utils: logger,
        utils: {
            config: {s3Settings},
        },
    } = req;

    if (!host) {
        throw new Error('Host in headers is undefined');
    }

    if (typeof fileName !== 'string') {
        throw new FileNameError(fileName);
    }

    if (!orderId) {
        throw new AppError('HOTELS_BUSINESS_TRIP_DOC_EMPTY_ORDER_ID', {
            code: 404,
            body: {error: {code: 'UNKNOWN_ORDER_ID'}},
        });
    }

    if (typeof orderId !== 'string') {
        throw new OrderIdError(orderId);
    }

    res.status(200).send();

    const printPagePath = getHotelsBusinessTripDocUrl(host, {orderId});

    try {
        logger.log('HOTELS_BUSINESS_TRIP_DOC_GENERATE_START', {orderId});

        const id = crypto.randomBytes(16).toString('hex');
        const localFileName = `${id}.pdf`;

        const pdfFileFsPath = await puppeteerClient({
            fileName: localFileName,
            orderId,
            printPagePath,
            logger,
            logEventsPrefix: 'HOTELS_BUSINESS_TRIP_DOC',
        });

        await saveFileToS3({
            fileContent: fs.readFileSync(pdfFileFsPath),
            fileName,
            orderId,
            s3Settings,
            logger,
        });

        fs.unlinkSync(pdfFileFsPath);

        logger.log('HOTELS_BUSINESS_TRIP_DOC_GENERATE_SUCCESS', {orderId});
    } catch (err) {
        logger.logError('HOTELS_BUSINESS_TRIP_DOC_GENERATE_FAILURE', {
            orderId,
            err,
        });
    }
}
