import fs from 'fs';

import {getHotelsBookVoucherUrl} from 'constants/urls';

import TRequest from 'server/types/TRequest';
import TResponse from 'server/types/TResponse';

import waitYandexMapItem from 'server/utilities/checkYandexMapLoaded';
import AppError from 'server/utilities/AppError/AppError';
import {FileNameError, OrderIdError} from 'server/utilities/AppError/errors';

import puppeteerClient from 'server/providers/puppeteerClient/puppeteerClient';
import saveFileToS3 from 'server/providers/s3Client/saveFileToS3';

export default async function generateHotelVoucher(
    req: TRequest,
    res: TResponse,
): Promise<void> {
    const {
        body: {orderId, fileName},
        headers: {host},
        utils: logger,
        utils: {
            config: {s3Settings},
        },
    } = req;

    if (!host) {
        throw new Error('Host in headers is undefined');
    }

    if (typeof fileName !== 'string') {
        throw new FileNameError(fileName);
    }

    if (!orderId) {
        throw new AppError('HOTELS_VOUCHER_EMPTY_ORDER_ID', {
            code: 404,
            body: {error: {code: 'UNKNOWN_ORDER_ID'}},
        });
    }

    if (typeof orderId !== 'string') {
        throw new OrderIdError(orderId);
    }

    res.status(200).send();

    const printPagePath = getHotelsBookVoucherUrl(host, {orderId});

    try {
        logger.log('HOTELS_VOUCHER_GENERATE_START', {orderId});

        const pdfFileFsPath = await puppeteerClient({
            waitItems: [waitYandexMapItem],
            fileName,
            orderId,
            printPagePath,
            logger,
            logEventsPrefix: 'HOTELS_VOUCHER',
        });

        await saveFileToS3({
            fileContent: fs.readFileSync(pdfFileFsPath),
            fileName,
            orderId,
            s3Settings,
            logger,
        });

        fs.unlinkSync(pdfFileFsPath);

        logger.log('HOTELS_VOUCHER_GENERATE_SUCCESS', {orderId});
    } catch (err) {
        logger.logError('HOTELS_VOUCHER_GENERATE_FAILURE', {orderId, err});
    }
}
