import fs from 'fs';

import {getHotelsBookVoucherUrl} from 'constants/urls';
import {EProjectName} from 'server/constants/EProjectName';

import TRequest from 'server/types/TRequest';
import TResponse from 'server/types/TResponse';

import waitYandexMapItem from 'server/utilities/checkYandexMapLoaded';
import AppError from 'server/utilities/AppError/AppError';
import {
    FileNameError,
    OrderIdError,
    GenerateError,
} from 'server/utilities/AppError/errors';
import buildErrorFromUnknown from 'server/utilities/AppError/buildErrorFromUnknown';

import puppeteerClient from 'server/providers/puppeteerClient/puppeteerClient';
import saveFileToS3 from 'server/providers/s3Client/saveFileToS3';

export default async function generateVoucher(
    req: TRequest,
    res: TResponse,
): Promise<void> {
    const {
        query: {orderId, fileName},
        headers: {host},
        utils: logger,
        utils: {
            config: {s3Settings},
        },
    } = req;

    if (!host) {
        throw new Error('Host in headers is undefined');
    }

    if (typeof fileName !== 'string') {
        throw new FileNameError(fileName);
    }

    if (!orderId) {
        throw new AppError('HOTELS_VOUCHER_EMPTY_ORDER_ID', {
            code: 404,
            body: {error: {code: 'UNKNOWN_ORDER_ID'}},
        });
    }

    if (typeof orderId !== 'string') {
        throw new OrderIdError(orderId);
    }

    const printPagePath = getHotelsBookVoucherUrl(host, {orderId});

    try {
        logger.log('HOTELS_VOUCHER_GENERATE_START', {orderId});

        const pdfFileFsPath = await puppeteerClient({
            waitItems: [waitYandexMapItem],
            fileName,
            orderId,
            printPagePath,
            logger,
            logEventsPrefix: 'HOTELS_VOUCHER',
        });
        const pdfFileContent = fs.readFileSync(pdfFileFsPath);
        const pdfFileS3Url = await saveFileToS3({
            fileContent: pdfFileContent,
            fileName,
            orderId,
            s3Settings,
            logger,
        });

        fs.unlinkSync(pdfFileFsPath);

        logger.log('HOTELS_VOUCHER_GENERATE_SUCCESS', {orderId});

        res.send({filePath: pdfFileS3Url});
    } catch (e) {
        throw new GenerateError(
            EProjectName.HOTELS,
            orderId,
            buildErrorFromUnknown(e),
        );
    }
}
