import TRequest from 'server/types/TRequest';
import TResponse from 'server/types/TResponse';

import {FileNameError} from 'server/utilities/AppError/errors';

import s3Client from '../../providers/s3Client/s3Client';

interface IS3ObjectError {
    $response: {
        body: {
            statusCode: number;
            statusMessage: string;
        };
    };
}

function isS3ObjectError(error: any): error is IS3ObjectError {
    return Boolean(
        error?.$response?.body?.statusCode &&
            error?.$response?.body?.statusMessage,
    );
}

export default async function state(
    req: TRequest,
    res: TResponse,
): Promise<void> {
    const {
        query: {fileName},
        utils: {
            config: {s3Settings},
        },
    } = req;

    if (typeof fileName !== 'string') {
        throw new FileNameError(fileName);
    }

    try {
        const {s3Host, s3BucketName} = s3Settings;

        const s3 = s3Client(s3Settings);

        const objectInfo = await s3.headObject({
            Bucket: s3BucketName,
            Key: fileName,
        });

        res.send({
            url: `${s3Host}/${s3BucketName}/${fileName}`,
            lastModified: objectInfo.LastModified,
        });
    } catch (error) {
        if (isS3ObjectError(error) && error.$response.body.statusCode === 404) {
            res.status(404).send({error: error.$response.body.statusMessage});
        } else {
            res.status(502).send(String(error));
        }
    }
}
