import first from 'lodash/first';

import {getBusesTicketUrl} from 'constants/urls';

import {CoreLogger} from '@yandex-data-ui/core/build/types';
import {EOrderRequestSnippet} from '../../../api/TravelApi/types/getAdminOrder/IGetAdminOrderRequest';

import waitYandexMapItem from '../../../utilities/checkYandexMapLoaded';

import TravelApi from '../../../api/TravelApi/TravelApi';
import getTempPdfLocation from './getTempPdfLocation';
import downloadAndSaveFile from './downloadAndSaveFile';
import puppeteerClient from '../../../providers/puppeteerClient/puppeteerClient';
import getPdfFilename from './getPdfFilename';
import getUniqPartnersTicketsUrls from './getUniqPartnersTicketsUrls';

export default async function fetchOrderAndWriteTicketsToDisk({
    baseFileName,
    orderId,
    travelApi,
    host,
    logger,
}: {
    baseFileName: string;
    orderId: string;
    travelApi: TravelApi;
    host: string;
    logger: CoreLogger;
}): Promise<string[]> {
    const order = await travelApi.getAdminOrder({
        id: orderId,
        snippets: EOrderRequestSnippet.S_PRIVATE_INFO,
    });

    const busOrderItem = first(order.busOrderItems);

    if (!busOrderItem) {
        throw new Error('No busOrderItem');
    }

    const ticketsPdfsPaths: string[] = [];

    const uniqPartnersTicketsUrls = getUniqPartnersTicketsUrls(busOrderItem);

    if (uniqPartnersTicketsUrls.length) {
        await Promise.all(
            uniqPartnersTicketsUrls.map((ticketUrl, index) => {
                const ticketPdfPath = getTempPdfLocation(baseFileName, index);

                ticketsPdfsPaths.push(ticketPdfPath);

                return downloadAndSaveFile(ticketUrl, ticketPdfPath);
            }),
        );
    } else {
        const {passengers, ride} = busOrderItem;

        const withMap = Boolean(
            ride.pointFrom.latitude && ride.pointFrom.longitude,
        );

        await Promise.all(
            passengers.map((passenger, index) => {
                const printPagePath = getBusesTicketUrl(host, {
                    orderId,
                    passengerIndex: index,
                });

                ticketsPdfsPaths.push(getTempPdfLocation(baseFileName, index));

                return puppeteerClient({
                    waitItems: withMap ? [waitYandexMapItem] : [],
                    fileName: getPdfFilename(baseFileName, index),
                    orderId,
                    printPagePath,
                    logger,
                    logEventsPrefix: 'BUSES_TICKETS',
                });
            }),
        );
    }

    return ticketsPdfsPaths;
}
