import puppeteer, {EvaluateFn, PDFOptions, WaitForOptions} from 'puppeteer';

import {CoreLogger} from '@yandex-data-ui/core/build/types';

import buildErrorFromUnknown from 'server/utilities/AppError/buildErrorFromUnknown';

export interface IWaitItem {
    condition: EvaluateFn;
    errorMessage: string;
}

interface IPuppeteerClientParams {
    fileName: string;
    orderId: string;
    printPagePath: string;
    logger: CoreLogger;
    waitItems?: IWaitItem[];
    logEventsPrefix: string;
}

export default async function puppeteerClient({
    fileName,
    orderId,
    printPagePath,
    logger,
    waitItems = [],
    logEventsPrefix,
}: IPuppeteerClientParams): Promise<string> {
    const fsPath = `/tmp/${fileName}`;

    const puppeteerLaunchConfig = {
        ignoreHTTPSErrors: true,
    };

    const puppeteerGoToConfig: WaitForOptions = {
        timeout: 40000,
        waitUntil: 'networkidle2',
    };

    const puppeteerPdfConfig: PDFOptions = {
        path: fsPath,
        format: 'a4',
        margin: {top: '12px', bottom: '12px'},
    };

    logger.log(`${logEventsPrefix}_PUPPETEER_START`, {orderId});

    try {
        const browser = await puppeteer.launch(puppeteerLaunchConfig);
        const page = await browser.newPage();
        const res = await page.goto(printPagePath, puppeteerGoToConfig);
        const responseStatus = res?.status();

        if (responseStatus === 200) {
            for (const wait of waitItems) {
                const {condition, errorMessage} = wait;

                try {
                    await page.waitForFunction(condition);
                } catch (e) {
                    logger.logFatal(errorMessage, e as Error);

                    throw e;
                }
            }

            await page.waitForTimeout(2000);

            await page.pdf(puppeteerPdfConfig);

            await browser.close();

            logger.log(`${logEventsPrefix}_PUPPETEER_SUCCESS`, {orderId});

            return fsPath;
        }

        throw new Error(`${logEventsPrefix}_PUPPETEER_PRINT_PAGE_ERROR`);
    } catch (e) {
        logger.logError(
            `${logEventsPrefix}_PUPPETEER_ERROR`,
            buildErrorFromUnknown(e),
            {
                orderId,
            },
        );

        throw new Error(`${logEventsPrefix}_PUPPETEER_ERROR`);
    }
}
