/* global __DEV__ */

import React from 'react';
import {Store} from 'redux';
import {renderToString} from 'react-dom/server';
import {StaticRouter as Router} from 'react-router-dom';
import {Provider} from 'react-redux';
import {StaticRouterContext} from 'react-router';
import 'moment-timezone';

import TRequest from './types/TRequest';
import TResponse from './types/TResponse';

import {IStoreInterface} from 'reducers/rootReducer';

import {setDateLocale} from 'utilities/dateUtils';
import locales from '../src/utilities/dateUtils/locales';
import createExtractor from './utilities/createExtractor';
import buildErrorFromUnknown from './utilities/AppError/buildErrorFromUnknown';

import App from 'containers/App/App';

import HtmlContainer from './render/components/HtmlContainer/HtmlContainer';

import {ServerFetchDataProvider} from 'contexts/ServerFetchDataContext';

import createReduxStore from '../src/redux/createReduxStore';
import fetchProjectReduxInfo, {
    TFetcher,
} from './redux/common/fetchProjectReduxInfo';

const SCRIPT_PROPS = {
    crossOrigin: 'anonymous',
};

interface IServerAppProps {
    req: TRequest;
    reduxStore: Store<IStoreInterface>;
    routerContext: StaticRouterContext;
}

const ServerApp: React.FC<IServerAppProps> = props => {
    const {req, reduxStore, routerContext} = props;

    return (
        <Provider store={reduxStore}>
            <Router location={req.url} context={routerContext}>
                <App />
            </Router>
        </Provider>
    );
};

export default () => async (req: TRequest, res: TResponse): Promise<void> => {
    try {
        const extractor = createExtractor();

        const fetchDataList: TFetcher[] = [];
        const reduxStore = createReduxStore();

        setDateLocale(locales.RU);

        const jsx = extractor.collectChunks(
            <ServerFetchDataProvider fetchDataList={fetchDataList}>
                <ServerApp
                    req={req}
                    reduxStore={reduxStore}
                    routerContext={{}}
                />
            </ServerFetchDataProvider>,
        );

        /* Fill reduxAsyncServerActions and reactLoadableModules */
        renderToString(jsx);

        const scriptElements = extractor.getScriptElements(SCRIPT_PROPS);
        const linkElements = extractor.getLinkElements(SCRIPT_PROPS);
        const styleElements = extractor.getStyleElements(SCRIPT_PROPS);

        await fetchProjectReduxInfo(req, reduxStore, fetchDataList);

        const reduxState = reduxStore.getState();
        const routerContext: StaticRouterContext = {};
        const html =
            '<!doctype html>' +
            renderToString(
                <HtmlContainer
                    styleElements={styleElements}
                    scriptElements={scriptElements}
                    linkElements={linkElements}
                    reduxState={reduxState}
                    nonce={req.nonce}
                >
                    <ServerApp
                        req={req}
                        reduxStore={reduxStore}
                        routerContext={routerContext}
                    />
                </HtmlContainer>,
            );

        if (routerContext.statusCode) {
            res.status(routerContext.statusCode);
        }

        res.send(html);
    } catch (e) {
        const error = buildErrorFromUnknown(e);

        req.utils.logError('An error occurred while rendering', e);

        if (__DEV__) {
            res.status(500).send(`
                <!doctype html>
                <html lang='ru'>
                    <head>
                        <title>Error</title>
                    </head>
                    <body>
                        <pre>${error.stack || error.message || error}</pre>
                    </body>
                </html>
            `);

            return;
        }

        res.status(500).send();
    }
};
