import React from 'react';

import {IWithClassName} from 'types/IWithClassName';

import cx from './Text.scss';

export interface ITextProps extends IWithClassName {
    /**
     * Вес текста
     *
     * @default 'normal'
     */
    weight?: TTextWeight;
    /**
     * Размер текста
     *
     * @default 'm'
     */
    size?: TTextSize;
    /**
     * Цвет текста
     *
     * @default 'primary'
     */
    color?: TTextColor;
    whiteSpace?: TWhiteSpace;
    onClick?: () => void;
    tag?: keyof JSX.IntrinsicElements;
    tagProps?: React.HTMLAttributes<Element>;
}

const Text: React.FC<ITextProps> = ({
    className,
    weight = 'normal',
    size = 'm',
    color = 'primary',
    whiteSpace,
    onClick,
    children,
    tag,
    tagProps,
}) => {
    const Tag = tag || 'span';

    return (
        <Tag
            onClick={onClick}
            {...tagProps}
            className={cx(
                className,
                textSizeCn(size),
                textWeightCn(weight),
                textColorCn(color),
                whiteSpace ? whiteSpaceCn(whiteSpace) : undefined,
            )}
        >
            {children}
        </Tag>
    );
};

export default Text;

export type TTextColor =
    | 'current'
    | 'primary'
    | 'secondary'
    | 'base-light'
    | 'link'
    | 'link-hover'
    | 'link-secondary'
    | 'success'
    | 'highlight'
    | 'alert';

export type TTextSize = 'xs' | 's' | 's-inset' | 'm' | 'l' | 'xl' | 'xxl';

export type TTextWeight = 'normal' | 'medium' | 'bold';

export type TWhiteSpace = 'normal' | 'nowrap' | 'pre-wrap';

function textSizeCn(size: TTextSize): string {
    return cx(`size_${size}`);
}

function textWeightCn(weight: TTextWeight): string {
    return cx(`weight_${weight}`);
}

function textColorCn(color: TTextColor): string {
    return cx(`color_${color}`);
}

function whiteSpaceCn(whitePace: TWhiteSpace): string {
    return cx(`whiteSpace_${whitePace}`);
}
