import React, {useCallback, useMemo} from 'react';
import {
    YMaps as YandexMaps,
    Map as YandexMap,
    Placemark as YandexMapPlacemark,
    ZoomControl,
    YMapsProps,
} from 'react-yandex-maps';

import {IWithClassName} from 'types/IWithClassName';
import ICoordinates from './types/ICoordinates';

import cx from './YandexMaps.scss';

const YANDEX_MAP_DEFAULT_CENTER: ICoordinates = {
    lat: 55.75,
    lng: 37.57,
};

const YANDEX_MAP_DEFAULT_ZOOM = 13;

const PROVIDER_QUERY: YMapsProps['query'] = {
    ns: 'use-load-option',
    load: 'Map,Placemark,control.FullscreenControl,geoObject.addon.balloon',
};

interface IYandexMapsComponent extends IWithClassName {
    defaultCenter?: ICoordinates;
    defaultZoom?: number;
    providerQuery?: YMapsProps['query'];
    mapMarkers?: ICoordinates[];
    canRenderZoomControl?: boolean;
}

const YandexMapsComponent: React.FC<IYandexMapsComponent> = props => {
    const {
        className,
        defaultCenter = YANDEX_MAP_DEFAULT_CENTER,
        defaultZoom = YANDEX_MAP_DEFAULT_ZOOM,
        providerQuery = PROVIDER_QUERY,
        mapMarkers = [],
        canRenderZoomControl = true,
        children,
    } = props;

    const handleLoad = useCallback(() => {
        document.body.classList.add('yandexMapLoaded');
    }, []);

    const markers = useMemo(() => {
        const canRenderMapMarkers = mapMarkers.length > 0;

        if (canRenderMapMarkers) {
            return mapMarkers.map(({lat, lng, ...markerProps}) => (
                <YandexMapPlacemark
                    key={`${lat}-${lng}`}
                    defaultGeometry={[lat, lng]}
                    {...markerProps}
                />
            ));
        }

        return children;
    }, [mapMarkers, children]);

    return (
        <YandexMaps query={providerQuery}>
            <YandexMap
                className={cx('map', className)}
                onLoad={handleLoad}
                defaultState={{
                    center: [defaultCenter.lat, defaultCenter.lng],
                    zoom: defaultZoom,
                }}
            >
                {canRenderZoomControl && <ZoomControl />}
                {markers}
            </YandexMap>
        </YandexMaps>
    );
};

export {YandexMaps, YandexMap, YandexMapPlacemark, ZoomControl};

export default YandexMapsComponent;
