import React from 'react';
import {RouteProps} from 'react-router-dom';
import {connect} from 'react-redux';
import {parse} from 'query-string';

import orderSelector from 'selectors/order/orderSelector';

import * as i18n from 'i18n/bus-Ticket';

import useCanRenderClientComponents from 'hooks/useCanRenderClientComponents';

import Layout from 'components/Layout/Layout';
import Text from 'components/Text/Text';
import Separator from 'components/Separator/Separator';
import YandexMaps from 'components/YandexMaps/YandexMaps';
import RouteInfo from 'projects/buses/pages/Ticket/components/RouteInfo/RouteInfo';
import Passenger from 'projects/buses/pages/Ticket/components/Passenger/Passenger';

import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';

import fetchOrderInfo from '../../../../../server/redux/common/fetchOrderInfo';

import cx from './Ticket.scss';

interface ITicketProps extends RouteProps, ReturnType<typeof orderSelector> {}

const Ticket: React.FC<ITicketProps> = props => {
    const {order, location} = props;

    const canRenderClientComponents = useCanRenderClientComponents();

    if (!order || !location) {
        return null;
    }

    const {passengerIndex} = parse(location.search);

    if (typeof passengerIndex !== 'string') {
        return null;
    }

    const busOrderItem = order.busOrderItems[0];
    const passenger = busOrderItem.passengers[Number(passengerIndex)];

    const {
        ride: {
            routeName,
            departureTime,
            arrivalTime,
            titlePointFrom,
            pointFrom,
            titlePointTo,
            pointTo,
            refundConditions,
        },
    } = busOrderItem;

    const coordinates =
        pointFrom.latitude && pointFrom.longitude
            ? {
                  lat: Number(pointFrom.latitude),
                  lng: Number(pointFrom.longitude),
              }
            : null;

    return (
        <Layout supportPhone="8 800 511-71-04">
            <div className={cx('content')}>
                <div className={cx('headerInfo')}>
                    <Text tag="div" color="secondary">
                        {i18n.orderNumber({number: order.yandexOrderId})}
                    </Text>
                </div>

                <div className={cx('ticket')}>
                    <RouteInfo
                        departureTime={departureTime}
                        arrivalTime={arrivalTime}
                        titlePointFrom={titlePointFrom}
                        titlePointTo={titlePointTo}
                        pointFrom={pointFrom}
                        pointTo={pointTo}
                        routeName={routeName}
                    />

                    <Separator borderWidth={2} margin={10} />

                    <Passenger passenger={passenger} />
                </div>

                {coordinates && (
                    <div className={cx('startStation')}>
                        <Text tag="div" size="xxl">
                            {titlePointFrom.title ||
                                titlePointFrom.supplierDescription}
                        </Text>

                        <Text tag="div">
                            {pointFrom.title || pointFrom.supplierDescription}
                        </Text>

                        {canRenderClientComponents && (
                            <div className={cx('map')}>
                                <YandexMaps
                                    defaultCenter={coordinates}
                                    defaultZoom={16}
                                    canRenderZoomControl={false}
                                    mapMarkers={[coordinates]}
                                />
                            </div>
                        )}
                    </div>
                )}

                <div className={cx('refundConditions')}>
                    <Text tag="div" color="secondary" weight="bold">
                        {i18n.refundConditionsTitle()}
                    </Text>

                    <Text tag="div" color="secondary">
                        {refundConditions}
                    </Text>
                </div>
            </div>
        </Layout>
    );
};

export default serverFetchDataDispatcher([fetchOrderInfo])(
    connect(orderSelector)(Ticket),
);
