import React, {useMemo} from 'react';

import {IWithClassName} from 'types/IWithClassName';
import IImage from 'server/api/TravelApi/types/getHotelOrder/IGetHotelOrderResponse/types/IHotelOfferInfo/IImage';
import ICoordinates from 'server/api/TravelApi/types/getHotelOrder/IGetHotelOrderResponse/types/IHotelOfferInfo/ICoordinates';
import IMapCoordinates from 'components/YandexMaps/types/ICoordinates';

import getHotelNameWithStars from 'projects/hotels/utilities/getHotelNameWithStars';

import * as i18nBookPayPage from 'i18n/hotels-BookPayPage';
import * as i18n from 'i18n/hotels-BookVoucherPDF';

import useCanRenderClientComponents from 'hooks/useCanRenderClientComponents';

import YandexMaps from 'components/YandexMaps/YandexMaps';

import cx from './OrderBaseHotelInfo.scss';

interface IOrderBaseHotelInfoProps extends IWithClassName {
    name: string;
    partnerHotelName: string;
    address: string;
    phone: string;
    stars: number | undefined;
    rating: number;
    images: IImage[];
    coordinates: ICoordinates;
    confirmationId: string;
}

const OrderBaseHotelInfo: React.FC<IOrderBaseHotelInfoProps> = props => {
    const {
        className,
        partnerHotelName,
        name,
        stars,
        address,
        phone,
        confirmationId,
        coordinates,
    } = props;

    const preparedHotelName = getHotelNameWithStars(name, stars);

    const canRenderClientComponents = useCanRenderClientComponents();

    const mapCoordinates = useMemo((): IMapCoordinates => {
        const {latitude, longitude} = coordinates;

        return {
            lat: latitude,
            lng: longitude,
        };
    }, [coordinates]);

    return (
        <div className={cx('orderBaseHotelInfo', className)}>
            <div className={cx('hotelInfo')}>
                <div className={cx('hotelInfoLeftColumn')}>
                    <h3 className={cx('hotelName')}>{preparedHotelName}</h3>
                    {address && (
                        <div className={cx('hotelInfoSection')}>
                            {i18n.hotelAddress()}
                            <span className={cx('hotelInfoSectionValue')}>
                                {address}
                            </span>
                        </div>
                    )}
                    {phone && (
                        <div className={cx('hotelInfoSection')}>
                            {i18n.hotelPhone()}
                            <a
                                className={cx('hotelInfoSectionValue')}
                                href={`tel:${phone}`}
                            >
                                {phone}
                            </a>
                        </div>
                    )}
                    {confirmationId && (
                        <div className={cx('hotelInfoSection')}>
                            {i18n.hotelOrderIdLabel()}
                            <span className={cx('hotelInfoSectionValue')}>
                                {confirmationId}
                            </span>
                        </div>
                    )}
                </div>
            </div>
            {canRenderClientComponents && (
                <div className={cx('map')}>
                    <YandexMaps
                        defaultCenter={mapCoordinates}
                        defaultZoom={16}
                        canRenderZoomControl={false}
                        mapMarkers={[mapCoordinates]}
                    />
                </div>
            )}
            <div className={cx('partner')}>
                {Boolean(partnerHotelName) &&
                    i18nBookPayPage.partnerHotelName({
                        hotelName: partnerHotelName,
                    })}
            </div>
        </div>
    );
};

export default OrderBaseHotelInfo;
