import React, {Component} from 'react';

import {IWithClassName} from 'types/IWithClassName';
import ICheckin from 'server/api/TravelApi/types/getHotelOrder/IGetHotelOrderResponse/types/IHotelOfferInfo/IPartnerHotelInfo/ICheckin';
import ICheckout from 'server/api/TravelApi/types/getHotelOrder/IGetHotelOrderResponse/types/IHotelOfferInfo/IPartnerHotelInfo/ICheckout';
import IHotelSearchRequestInfo from 'server/api/TravelApi/types/getHotelOrder/IGetHotelOrderResponse/types/IHotelOfferInfo/IHotelSearchRequestInfo';
import {IOrderGuestsInfoResponse} from 'server/api/TravelApi/types/getHotelOrder/IGetHotelOrderResponse/IGetHotelOrderResponse';

import getHotelCheckDates from 'projects/hotels/utilities/getHotelCheckDates/getHotelCheckDates';

import * as i18n from 'i18n/hotels-BookVoucherPDF';

import cx from './OrderBookInfo.scss';

interface IOrderBookInfoProps extends IWithClassName {
    checkin: ICheckin;
    checkout: ICheckout;
    guestInfo: IOrderGuestsInfoResponse;
    requestInfo: IHotelSearchRequestInfo;
}

class OrderBookInfo extends Component<IOrderBookInfoProps> {
    renderHotelCheckDate(
        title: string,
        value: string,
        totalNights?: number,
    ): React.ReactNode {
        return (
            <div className={cx('checkDate')}>
                <h5 className={cx('checkDateTitle')}>{title}</h5>
                <div className={cx('checkDateValue')}>
                    {value}
                    {totalNights && (
                        <span className={cx('checkDateNights')}>
                            {i18n.nights({totalNights})}
                        </span>
                    )}
                </div>
            </div>
        );
    }

    renderHotelCheckDates(): React.ReactNode {
        const {checkin, checkout, requestInfo} = this.props;

        const {checkinData, checkoutData, totalNights} = getHotelCheckDates({
            requestInfo,
            checkin,
            checkout,
        });

        return (
            <>
                {this.renderHotelCheckDate(
                    checkinData.title,
                    checkinData.value,
                    totalNights,
                )}
                {this.renderHotelCheckDate(
                    checkoutData.title,
                    checkoutData.value,
                )}
            </>
        );
    }

    renderGuest(
        guestName: string,
        index: number,
        guests: string[],
    ): React.ReactNode {
        const canRenderComma = index + 1 !== guests.length;

        return (
            <span className={cx('guest')} key={index}>
                {`${guestName}${canRenderComma ? ', ' : ''}`}
            </span>
        );
    }

    renderGuestsInfo(): React.ReactNode {
        const {
            guestInfo: {guests},
        } = this.props;

        const preparedGuests = guests
            .filter(({firstName, lastName}) => firstName || lastName)
            .map(({firstName, lastName}) =>
                [firstName, lastName].filter(Boolean).join(' '),
            );

        return (
            <div className={cx('guests')}>
                <h5 className={cx('guestsTitle')}>{i18n.guestsTitle()}</h5>
                {this.renderGuestsCount()}
                {preparedGuests.map((guestName, index, array) =>
                    this.renderGuest(guestName, index, array),
                )}
            </div>
        );
    }

    renderGuestsCount(): React.ReactNode {
        const {
            guestInfo: {numAdults, childAges},
        } = this.props;

        if (!numAdults) {
            return null;
        }

        const parts = [i18n.adultsCount({count: numAdults})];

        if (childAges?.length > 0) {
            const exceptLastAgeStrings = childAges
                .slice(0, childAges.length - 1)
                .map(count => i18n.age({count}));
            const lastChildAgeString = i18n.age({
                count: childAges[childAges.length - 1],
            });

            parts.push(
                i18n.childrenCount({
                    count: childAges.length,
                    childAges: [
                        exceptLastAgeStrings.join(', '),
                        lastChildAgeString,
                    ]
                        .filter(Boolean)
                        .join(` ${i18n.and()} `),
                }),
            );
        }

        return <div className={cx('guest')}>{parts.join(', ')}</div>;
    }

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div className={cx('orderBookInfo', className)}>
                <div className={cx('leftColumn')}>
                    {this.renderHotelCheckDates()}
                    {this.renderGuestsInfo()}
                </div>
            </div>
        );
    }
}

export default OrderBookInfo;
