import React, {Component} from 'react';
import {connect} from 'react-redux';
import {RouteProps} from 'react-router-dom';

import EHotelOrderStatus from 'server/api/TravelApi/types/getHotelOrder/IGetHotelOrderResponse/types/EHotelOrderStatus';

import hotelOrderSelector from 'selectors/hotels/hotelOrderSelector';

import getSupportPhone from 'projects/hotels/utilities/getSupportPhone';

import * as i18n from 'i18n/hotels-BookVoucherPDF';

import OrderBaseHotelInfo from 'projects/hotels/components/OrderBaseHotelInfo/OrderBaseHotelInfo';
import OrderBookInfo from 'projects/hotels/components/OrderBookInfo/OrderBookInfo';
import OrderRoomInfo from 'projects/hotels/components/OrderRoomInfo/OrderRoomInfo';
import RouteWithStatusCode from 'components/RouteWithStatusCode/RouteWithStatusCode';
import Layout from 'components/Layout/Layout';

import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';

import fetchOrderInfo from 'server/redux/hotels/fetchOrderInfo';

import cx from './BookVoucher.scss';

interface IBookVoucherProps
    extends RouteProps,
        ReturnType<typeof hotelOrderSelector> {}

class BookVoucher extends Component<IBookVoucherProps> {
    checkConfirmedOrderStatus(): boolean {
        const {order} = this.props;

        return order?.status === EHotelOrderStatus.CONFIRMED;
    }

    getSupportPhone(): string | null {
        const {order} = this.props;

        if (!order) {
            return null;
        }

        const {
            orderInfo: {
                basicHotelInfo: {locationType},
            },
        } = order;

        return getSupportPhone(locationType);
    }

    renderOrderInfo(): React.ReactNode {
        const {order} = this.props;

        if (!order) {
            return null;
        }

        const {yandexOrderId} = order;

        return (
            <div className={cx('orderInfo')}>
                <div className={cx('orderInfoSection')}>
                    <h3 className={cx('orderInfoTitle')}>{i18n.bookTitle()}</h3>
                    {yandexOrderId && (
                        <div className={cx('orderId')}>
                            {i18n.yandexOrderIdLabel()}
                            <div className={cx('orderIdValue')}>
                                {yandexOrderId}
                            </div>
                        </div>
                    )}
                </div>
                {this.renderBaseHotelInfo()}
                {this.renderOrderBookInfo()}
                {this.renderOrderRoomInfo()}
            </div>
        );
    }

    /* Render left column blocks */

    renderBaseHotelInfo(): React.ReactNode {
        const {order} = this.props;

        if (!order) {
            return null;
        }

        const {
            orderInfo: {basicHotelInfo, partnerHotelInfo},
            confirmationInfo: {confirmationId},
        } = order;

        return (
            <OrderBaseHotelInfo
                className={cx('baseHotelInfo')}
                name={basicHotelInfo.name}
                partnerHotelName={partnerHotelInfo.name}
                address={basicHotelInfo.address}
                phone={basicHotelInfo.phone}
                stars={basicHotelInfo.stars}
                rating={basicHotelInfo.rating}
                images={partnerHotelInfo.images}
                coordinates={basicHotelInfo.coordinates}
                confirmationId={confirmationId}
            />
        );
    }

    renderOrderBookInfo(): React.ReactNode {
        const {order} = this.props;

        if (!order) {
            return null;
        }

        const {
            orderInfo: {
                requestInfo,
                partnerHotelInfo: {checkin, checkout},
            },
            guestInfo,
        } = order;

        return (
            <OrderBookInfo
                className={cx('orderBaseInfo')}
                checkin={checkin}
                checkout={checkout}
                guestInfo={guestInfo}
                requestInfo={requestInfo}
            />
        );
    }

    renderOrderRoomInfo(): React.ReactNode {
        const {order} = this.props;

        if (!order) {
            return null;
        }

        const {
            orderInfo: {
                partnerRoomInfo,
                pansionInfo,
                partnerHotelInfo: {
                    checkin: {instructions, specialInstructions},
                    fees,
                    policies: {knowBeforeYouGo},
                },
                bedGroups,
            },
        } = order;

        return (
            <OrderRoomInfo
                className={cx('orderRoomInfo')}
                roomInfo={partnerRoomInfo}
                bedGroups={bedGroups}
                pansionInfo={pansionInfo}
                instructions={instructions}
                specialInstructions={specialInstructions}
                fees={fees}
                knowBeforeYouGo={knowBeforeYouGo}
            />
        );
    }

    render(): React.ReactNode {
        const canRenderOrderInfo = this.checkConfirmedOrderStatus();

        if (canRenderOrderInfo) {
            const supportPhone = this.getSupportPhone();

            return (
                <Layout supportPhone={supportPhone}>
                    {this.renderOrderInfo()}
                </Layout>
            );
        }

        return <RouteWithStatusCode statusCode={502} />;
    }
}

export default serverFetchDataDispatcher([fetchOrderInfo])(
    connect(hotelOrderSelector)(BookVoucher),
);
