import React, {useCallback, useState} from 'react';
import moment from 'moment';

import {EProgress} from './types/EProgress';
import {IWithClassName} from 'types/IWithClassName';

import {delay} from 'utilities/async/delay';

import cx from './DocumentGenerator.scss';

interface IDocumentGeneratorProps extends IWithClassName {
    origin: string;

    title: string;

    exampleOrderId: string;
    exampleFileName: string;

    apiGeneratePath: string;

    passengerIndex?: number;

    getViewUrl(
        origin: string,
        params: {orderId: string; passengerIndex?: number},
    ): string;
}

const DocumentGenerator: React.FC<IDocumentGeneratorProps> = props => {
    const {
        className,
        origin,
        title,
        exampleOrderId,
        exampleFileName,
        apiGeneratePath,
        passengerIndex,
        getViewUrl,
    } = props;

    const [orderId, setOrderId] = useState(exampleOrderId);
    const [fileName, setFileName] = useState(exampleFileName);

    const [state, setState] = useState(EProgress.INITIAL);

    const [documentUrl, setDocumentUrl] = useState('');

    const handleGenerate = useCallback(async () => {
        setState(EProgress.LOADING);

        try {
            const requestedMoment = moment();

            await fetch(apiGeneratePath, {
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    orderId,
                    fileName,
                }),
                method: 'POST',
            });

            for (let i = 0; i < 5; i++) {
                await delay(3 * 1000);

                const params = new URLSearchParams({
                    fileName,
                });

                const result = await fetch(`/pdf/state?${params.toString()}`, {
                    method: 'GET',
                });

                const body = await result.json();

                if (
                    body &&
                    body.url &&
                    requestedMoment.isBefore(body.lastModified)
                ) {
                    setDocumentUrl(body.url);

                    break;
                }
            }

            setState(EProgress.SUCCESS);
        } catch (e) {
            setState(EProgress.ERROR);
        }
    }, [apiGeneratePath, fileName, orderId]);

    return (
        <div className={className}>
            <h2>{title}</h2>
            <div className={cx('block')}>
                <label className={cx('label')}>ID заказа</label>
                <input
                    className={cx('input')}
                    placeholder="ID заказа"
                    value={orderId}
                    onChange={(e): void => setOrderId(e.target.value)}
                />
                <a
                    className={cx('link')}
                    href={getViewUrl(origin, {
                        orderId,
                        passengerIndex,
                    })}
                    target="_blank"
                    rel="noreferrer"
                >
                    Посмотреть документ
                </a>
            </div>
            <div className={cx('block')}>
                <label className={cx('label')}>Имя файла</label>
                <input
                    className={cx('input')}
                    placeholder="Имя файла"
                    value={fileName}
                    onChange={(e): void => setFileName(e.target.value)}
                />
                <button
                    className={cx('button')}
                    onClick={handleGenerate}
                    disabled={state === EProgress.LOADING}
                >
                    Сгенерировать документ
                </button>
                {state === EProgress.ERROR && (
                    <span className={cx('error')}>Error</span>
                )}
                {state === EProgress.SUCCESS && documentUrl && (
                    <a
                        className={cx('link')}
                        href={documentUrl}
                        target="_blank"
                        rel="noreferrer"
                    >
                        Скачать документ
                    </a>
                )}
            </div>
        </div>
    );
};

export default DocumentGenerator;
