import moment, {Moment} from 'moment';

import {DAY, EVENING, MORNING, NIGHT} from './constants';

import {TTimeOfDay} from 'utilities/dateUtils/types';

import {
    HUMAN_SHORT_WITH_YEAR,
    HUMAN_SHORT,
    HUMAN,
    HUMAN_WITH_YEAR,
} from './formats';

export type TDate = Moment;
/**
 * number - Количество миллисекунд, прошедших с 1 января 1970 года.
 * string - Любая строка, которая преобразуется в дату.
 */
export type TDateLike = number | string | Date | TDate;

/**
 * Устанавливает язык для дат.
 *
 * @param {string} locale - Язык.
 *
 * @return {string}
 */
export function setDateLocale(locale: string): string {
    return moment.locale(locale);
}

/**
 * Возвращает текущий язык.
 *
 * @return {string}
 */
export function getDateLocale(): string {
    return moment.locale();
}

/**
 * Возвращает количество миллисекунд прошедших с 1 января 1970 года до сегодняшнего дня.
 *
 * TODO: добавить коррекцию времени с сервером.
 * @see https://gitlab.yandex-team.ru/rasp/trains-front/blob/dev/common/lib/date/utils.js#L85
 *
 * @return {number}
 */
export function getNow(): number {
    return Date.now();
}

/**
 * Преобразование строки или нативного объекта Date в формат даты, с которым работает приложение.
 *
 * @param {TDateLike} date - Дата, которую нужно обработать.
 * @param {string} [format] - Формат даты, если нужно обработать строку.
 *
 * @return {TDate}
 */
export function parseDate(date: TDateLike, format?: string): TDate {
    return moment(date, format);
}

/**
 * Является ли дата валидной.
 *
 * @param {TDateLike} date - Дата, которую нужно проверить.
 *
 * @return {boolean}
 */
export function isValidDate(date: TDateLike): boolean {
    return moment(date).isValid();
}

/**
 * Отображение даты в необходимом формате.
 *
 * @param {TDateLike} date - Дата, которую нужно преобразовать в строку.
 * @param {string} format - Формат даты.
 *
 * @return {string}
 */
export function formatDate(date: TDateLike, format: string): string {
    return moment(date).format(format);
}

/**
 * Отображение даты в человекочитаемом виде.
 * Если год даты совпадает с текущим годом, то год не отображается.
 *
 * @param {TDateLike} date - Дата, которую нужно преобразовать в строку.
 *
 * @return {string}
 */
export function humanFormatDate(date: TDateLike): string {
    const isCurrentYear = moment(date).isSame(moment(), 'year');
    const yearIsShown = !isCurrentYear;

    return formatDate(date, yearIsShown ? HUMAN_WITH_YEAR : HUMAN).replace(
        '.',
        '',
    );
}

/**
 * Отображение диапазона времени в человекочитаемом виде.
 * Если дата окончания не задана, то будет выведена только дата начала.
 *
 * @param {TDateLike} from - Дата начала.
 * @param {TDateLike} [to] - Дата окончания.
 *
 * @return {string}
 */
export function formatDateRange(from: TDateLike, to?: TDateLike): string {
    if (!to) {
        return humanFormatDate(from);
    }

    const fromM = moment(from);
    const toM = moment(to);
    const isSameYear = fromM.isSame(toM, 'year');
    const isCurrentYear = fromM.isSame(moment(), 'year');
    const yearIsShown = !isSameYear || !isCurrentYear;

    const formattedFrom = formatDate(
        from,
        yearIsShown ? HUMAN_SHORT_WITH_YEAR : HUMAN_SHORT,
    ).replace('.', '');
    const formattedTo = formatDate(
        to,
        yearIsShown ? HUMAN_SHORT_WITH_YEAR : HUMAN_SHORT,
    ).replace('.', '');

    return `${formattedFrom} — ${formattedTo}`;
}

/**
 * Проверяем дату, чтобы бекенд корректно мог ее вставить в таблицу
 * Следующие требования: дата валидна, год 1700 и выше и год меньше 10000
 *
 * @param {TDateLike} date - Дата
 *
 * @return {boolean}
 */
export function isValidDateForBackend(date: TDateLike): boolean {
    if (!isValidDate(date)) {
        return false;
    }

    const momentDate = moment(date);
    const dateMin = moment('1700-01-01');
    const dateMax = moment('9999-12-31');

    return !momentDate.isBefore(dateMin) && !momentDate.isAfter(dateMax);
}

/**
 * Проверить, в будущем ли дата
 * Вернет true, если в будущем и false, если нет
 *
 * @param {TDateLike} date - дата, которую проверяем
 *
 * @return {boolean}
 */
export function isDateInFuture(date: TDateLike): boolean {
    return moment(date).isAfter(moment());
}

export function getTimeOfDay(hour: string | number): TTimeOfDay | void {
    hour = Number(hour);

    switch (true) {
        case hour >= 0 && hour < 6:
            return NIGHT;

        case hour >= 6 && hour < 12:
            return MORNING;

        case hour >= 12 && hour < 18:
            return DAY;

        case hour >= 18 && hour <= 23:
            return EVENING;
    }
}
