const path = require('path');
const webpack = require('webpack');
const {mergeWithRules, CustomizeRule} = require('webpack-merge');
const MiniCSSExtractPlugin = require('mini-css-extract-plugin');
const TerserPlugin = require('terser-webpack-plugin');
const CssMinimizerPlugin = require('css-minimizer-webpack-plugin');
const StatsWebpackWriterPlugin = require('stats-webpack-plugin');
const {BundleAnalyzerPlugin} = require('webpack-bundle-analyzer');
const LoadablePlugin = require('@loadable/webpack-plugin');

const uglifyJsPluginSettings = require('./utilities/getUglifyJsPluginSettings');
const getFileNameByBuildType = require('./utilities/getFileNameByBuildType');
const {
    srcPath,
    clientBuildPath,
    serverBuildPath,
    clientPublicPath,
    excludeNodeModulesRegExpPath,
} = require('./utilities/getPathParams');

const {
    isDevelopment,
    isProduction,
    isBuildBundleAnalyzer,
} = require('./variables');
const createCommonWebpackConfig = require('./common.webpack.config');

const commonWebpackConfig = createCommonWebpackConfig({isServer: false});

const smartStrategyMergeParams = {
    module: {
        rules: {
            test: CustomizeRule.Match,
            include: CustomizeRule.Match,
            exclude: CustomizeRule.Match,
            use: CustomizeRule.Prepend,
        },
        plugins: CustomizeRule.Prepend,
    },
};

const clientEntry = [path.join(srcPath, 'client.tsx')];
const clientMinimizerPlugins = [];
const clientWebpackPlugins = [
    new MiniCSSExtractPlugin({
        filename: getFileNameByBuildType({fileType: 'css', isProduction}),
        chunkFilename: getFileNameByBuildType({fileType: 'css', isProduction}),
    }),
    new webpack.NormalModuleReplacementPlugin(
        /server\/redux\/.+?/,
        'lodash/noop',
    ),
    new LoadablePlugin({
        writeToDisk: {
            filename: serverBuildPath,
        },
    }),
];

if (isDevelopment) {
    clientEntry.unshift('webpack-hot-middleware/client');
    clientWebpackPlugins.unshift(new webpack.HotModuleReplacementPlugin());
}

if (isProduction) {
    clientMinimizerPlugins.push(
        new TerserPlugin(uglifyJsPluginSettings),
        new CssMinimizerPlugin({
            minimizerOptions: {
                preset: [
                    'default',
                    {
                        discardComments: {removeAll: true},
                    },
                ],
            },
        }),
    );
    clientWebpackPlugins.push(new StatsWebpackWriterPlugin('stats.json'));
}

if (isBuildBundleAnalyzer) {
    clientWebpackPlugins.push(
        new BundleAnalyzerPlugin({
            analyzerMode: 'static',
            openAnalyzer: false,
            reportFilename: 'stats.html',
        }),
    );
}

const miniCssExtractLoader = MiniCSSExtractPlugin.loader;
const prependStylesLoaders = [
    'classnames-loader',
    {
        loader: miniCssExtractLoader,
        options: {
            esModule: false,
        },
    },
];

if (isDevelopment) {
    prependStylesLoaders.splice(1, 0, 'css-hot-loader');
}

const clientWebpackConfig = {
    name: 'client',
    devtool: isDevelopment ? 'cheap-module-source-map' : false,
    target: 'web',
    entry: {client: clientEntry},
    output: {
        path: clientBuildPath,
        publicPath: clientPublicPath,
        filename: getFileNameByBuildType({fileType: 'js', isProduction}),
        chunkFilename: getFileNameByBuildType({fileType: 'js', isProduction}),
    },
    module: {
        rules: [
            {
                test: /\.([tj])sx?$/,
                exclude: excludeNodeModulesRegExpPath,
                use: [
                    {
                        loader: 'babel-loader',
                        options: {
                            presets: [
                                [
                                    '@babel/preset-env',
                                    {
                                        corejs: 3,
                                        useBuiltIns: 'entry',
                                        modules: false,
                                    },
                                ],
                                '@babel/preset-react',
                                [
                                    '@babel/preset-typescript',
                                    {allowNamespaces: true},
                                ],
                            ],
                            plugins: ['@babel/plugin-syntax-dynamic-import'],
                        },
                    },
                ],
            },
            {
                test: /\.scss$/,
                exclude: excludeNodeModulesRegExpPath,
                use: prependStylesLoaders,
            },
            {
                test: /\.css$/,
                exclude: excludeNodeModulesRegExpPath,
                use: prependStylesLoaders,
            },
        ],
    },
    plugins: clientWebpackPlugins,
    optimization: {
        minimizer: clientMinimizerPlugins,
        splitChunks: {
            chunks: 'all',
            cacheGroups: {
                vendors: {
                    test: /node_modules[\\/]/,
                    name: 'vendors',
                    chunks: 'all',
                    priority: 0,
                },
                common: {
                    name: 'common',
                    chunks: 'async',
                    minChunks: 2,
                    minSize: 0,
                    priority: 0,
                },
            },
        },
    },
};

module.exports = mergeWithRules(smartStrategyMergeParams)(
    commonWebpackConfig,
    clientWebpackConfig,
);
