const path = require('path');
const webpack = require('webpack');
const CopyWebpackPlugin = require('copy-webpack-plugin');
const MomentLocalesPlugin = require('moment-locales-webpack-plugin');

const {
    srcPath,
    stylesPath,
    rootPath,
    i18nPath,
    excludeNodeModulesRegExpPath,
    publicBuildPath,
    publicSourcePath,
    nodeModulesPath,
    clientPublicPath,
} = require('./utilities/getPathParams');
const getGlobalVariables = require('./utilities/getGlobalVariables');

const {
    nodeEnv,
    buildLang,
    isDevelopment,
    isProduction,
} = require('./variables');

const IMAGES_REG_EX = /\.(ico|bmp|gif|jpe?g|png|svg)$/;

const scssLoaders = [
    {
        loader: 'css-loader',
        options: {
            url: false,
            import: false,
            sourceMap: true,
            esModule: false,
            modules: {
                localIdentName: isProduction
                    ? '[contenthash:base64:5]'
                    : '[folder]__[local]_[contenthash:base64:5]',
            },
        },
    },
    {
        loader: 'resolve-url-loader',
        options: {sourceMap: true},
    },
    {
        loader: 'postcss-loader',
        options: {sourceMap: true},
    },
    {
        loader: 'sass-loader',
        options: {
            implementation: require('sass'),
            sourceMap: true,
            sassOptions: {
                includePaths: [path.join(srcPath, 'styles')],
            },
        },
    },
    {
        loader: 'sass-resources-loader',
        options: {
            sourceMap: true,
            hoistUseStatements: true,
            resources: [
                path.join(stylesPath, 'mixins.scss'),
                path.join(stylesPath, 'variables.scss'),
            ],
        },
    },
];

const cssLoaders = ['css-loader'];

module.exports = ({isServer}) => {
    const commonWebpackPlugins = [
        new webpack.DefinePlugin({
            ...getGlobalVariables(isServer),
            'process.env.NODE_ENV': JSON.stringify(nodeEnv),
            'process.env.BEM_LANG': JSON.stringify(buildLang),
        }),
        new MomentLocalesPlugin({localesToKeep: ['ru']}),
    ];

    if (!isServer && !isDevelopment) {
        commonWebpackPlugins.unshift(
            new CopyWebpackPlugin({
                patterns: [
                    {
                        from: publicSourcePath,
                        to: publicBuildPath,
                    },
                ],
            }),
        );
    }

    return {
        mode: isDevelopment ? 'development' : 'production',
        context: srcPath,
        module: {
            rules: [
                {
                    test: /\.scss$/,
                    exclude: excludeNodeModulesRegExpPath,
                    use: scssLoaders,
                },
                {
                    test: /\.css$/,
                    exclude: excludeNodeModulesRegExpPath,
                    use: cssLoaders,
                },
                {
                    test: IMAGES_REG_EX,
                    include: /node_modules\/(@yandex-lego)/,
                    type: 'asset/resource',
                    generator: {
                        publicPath: clientPublicPath,
                    },
                },
                {
                    test: IMAGES_REG_EX,
                    exclude: /node_modules/,
                    type: 'asset/resource',
                    generator: {
                        publicPath: clientPublicPath,
                    },
                },
                {
                    test: /\.(ttf|eot|woff2?)$/,
                    exclude: excludeNodeModulesRegExpPath,
                    use: ['noop'],
                },
            ],
        },
        plugins: commonWebpackPlugins,
        resolve: {
            alias: {
                sagas: path.join(srcPath, 'redux', 'sagas'),
                reducers: path.join(srcPath, 'redux', 'reducers'),
                selectors: path.join(srcPath, 'redux', 'selectors'),
                components: path.join(srcPath, 'components'),
                projects: path.join(srcPath, 'projects'),
                icons: path.join(srcPath, 'icons'),
                containers: path.join(srcPath, 'containers'),
                utilities: path.join(srcPath, 'utilities'),
                hooks: path.join(srcPath, 'hooks'),
                constants: path.join(srcPath, 'constants'),
                contexts: path.join(srcPath, 'contexts'),
                i18n: i18nPath,
                types: path.join(srcPath, 'types'),
                server: path.join(rootPath, 'server'),
                'moment-timezone': path.join(
                    nodeModulesPath,
                    'moment-timezone',
                    'builds',
                    'moment-timezone-with-data-2012-2022.min',
                ),
            },
            fallback: {url: false},
            extensions: ['.json', '.ts', '.tsx', '.jsx', '.js', '.scss'],
        },
        resolveLoader: {
            modules: [
                'node_modules',
                path.join(rootPath, 'webpack', 'loaders'),
            ],
        },
    };
};
