enum ERumErrorLevel {
    INFO = 'info',
    DEBUG = 'debug',
    WARN = 'warn',
    ERROR = 'error',
    FATAL = 'fatal',
}

interface IRumSubpage {
    readonly '2924': string;
    readonly '2925': number;
    readonly '689.2322': number;
    readonly cancel(): void;
    readonly isCanceled(): boolean;
}

interface IRumErrorOptions {
    message: string; // Текст ошибки
    block?: string; // Блок, в котором произошла ошибка
    method?: string; // Метод, в котором произошла ошибка
    source?: string; // Источник ошибки
    sourceMethod?: string; // Метод источника ошибки
    type?: string; // Тип ошибки, например: network, logic
    page?: string; // Страница сервиса, например: index, search, product
    service?: string; // Если на одной странице живет несколько сервисов, например: Эфир, Дзен на морде
    level: ERumErrorLevel; // !! Доступные значения: info, debug, warn, error, fatal
    additional?: Record<string, string | number | boolean>; // key-value дополнительной информации, без вложенности. Не нужно передавать сюда state приложения
}

interface IRum {
    /**
     * Создает инстанс подстраницы, чтобы отправлять счетчики в отдельном неймспейсе.
     *
     * @param {String} pageName - имя подстраницы
     * @param {String} [start] - время создания
     */
    makeSubPage(pageName: string, start?: number): IRumSubpage;
    /**
     * Отправить счетчик с временной меткой.
     *
     * @param {String} counterId - Код BlockStat. ID метки времени
     * @param {Number} [time] - Время. Если не передано, вычисляется время от начала навигации до момента вызова функции
     * @param {Boolean} [addPerfMark=true] - Добавлять метку в User Timing или нет, true по умолчанию
     * @param {Object} [params=null] - Кастомные параметры счетчика. Сюда же можно передать инстанс подстраницы
     */
    sendTimeMark(
        counterId: string,
        time: number,
        addPerfMark?: boolean,
        params?: IRumSubpage & Record<string, any>,
    ): void;
    /**
     * Отправить счетчик с дельтой времени.
     *
     * @param {String} counterId - Код BlockStat. ID метки времени
     * @param {Number} [delta] - Точное значение дельты
     * @param {Object} [subPage] - Инстанс подстраницы
     */
    sendDelta(counterId: string, delta?: number, subPage?: IRumSubpage): void;
    /**
     * Получить время от начала навигации.
     *
     * @returns {Number}
     */
    getTime(): number;
    /**
     * Отправляет событие об отрисовке основного содержимого
     */
    sendHeroElement(): void;
    /**
     * Отправить данные о трафике
     */
    sendTrafficData(): void;
    /**
     * Отправить текущее значение CLS и начать считать заново
     */
    finalizeLayoutShiftScore(): void;
    /**
     * Отправить текущее значение LCP и начать искать заново
     */
    finalizeLargestContentfulPaint(): void;
    /**
     * Переопределяет RUM параметры
     */
    setVars(vars: {
        // eslint-disable-next-line camelcase
        rum_id?: string;
        '-page'?: string;
    }): void;
    logError: (
        /** https://a.yandex-team.ru/arcadia/frontend/packages/rum-counter */
        options: IRumErrorOptions,
        error?: Error,
    ) => void;
    ERROR_LEVEL: {
        INFO: ERumErrorLevel.INFO;
        DEBUG: ERumErrorLevel.DEBUG;
        WARN: ERumErrorLevel.WARN;
        ERROR: ERumErrorLevel.ERROR;
        FATAL: ERumErrorLevel.FATAL;
    };
}

interface ICmntOptions {
    apiKey: string;
    entityId: string;
    tld: 'ru';
}

type TCmntTheme = 'serp' | 'mg' | 'dark' | 'dark2' | 'maps' | 'maps-dark';

interface ICmnt {
    theme: TCmntTheme;
    lang: 'ru' | 'be' | 'en' | 'kk' | 'tr' | 'uk' | 'uz';
    service: string;

    init?(elementId: string, options: ICmntOptions);
    update?(elementId: string, options: Pick<ICmntOptions, 'entityId'>);
    destroy?(elementId: string);

    getLoadedThemes?(): string[];
    loadTheme?(theme: TCmntTheme, callback: () => void);
}

interface IAdvManagerRenderParams {
    blockId: string;
    pageNumber?: number;
    renderTo: string;
    async: boolean;
    onRender(...args: any[]): void;
}

interface IChatWidgetConstructorParams {}

interface IChatWidget {
    new (params: IChatWidgetConstructorParams): IChatWidget;
    showChat(params: {guid: string; chatMetadata: object}): void;
}

interface IYa {
    Rum: IRum;
    Cmnt: ICmnt;
    Context: {
        AdvManager: {
            render(params: IAdvManagerRenderParams, cb?: () => void): void;
        };
    };
    adfoxCode: {
        create(params: IAdFoxBannerParams): void;
    };
    ChatWidget: IChatWidget;
}

interface IAppData {
    appEnv?: 'beta' | 'development' | 'testing' | 'production';
    jaegerDebugId?: string;
    uid: string;
}

interface IAdFoxBannerParams {
    ownerId: number;
    containerId: string;
    params: Record<string, string>;
    onLoad?: (data: {bundleName: string; bundleParams: object}) => void;
    onRender?: (data: {bundleName: string; bundleParams: object}) => void;
}

// Используем импорты такого вида, так как import {abc} from 'abc';
// Приводит к ошибке
// Ref: https://stackoverflow.com/a/51114250/1236038
type TRootReducerType =
    import('src/redux/reducers/rootReducer').TRootReducerType;
type TPreloadedState = import('redux').PreloadedState<TRootReducerType>;

// eslint-disable-next-line @typescript-eslint/naming-convention
interface Window {
    // Такой же id используется в переменной METRIKA_COUNTER_ID в constants/common.ts
    // Используется с id из-за пакета @yandex-data-ui/node-components/components/metrika
    yaCounter50912507?: IMetrika;
    yaCounter50912507Queue?: TMetrikaQueueItem[];
    yaContextCb?: (() => unknown)[];
    dataLayer?: any[];
    timeCorrection: number;
    Ya: IYa;
    appData?: IAppData;
    ym: (id: any, method: string, target: string, params: any) => void;
    /**
     * Класс который появляется в устройствах поддерживающих ApplePay
     */
    ApplePaySession?: {
        canMakePayments(): boolean;
    };
    yandexChatWidgetCallback: () => void;
    chatWidget: IChatWidget;
    chrome: object;

    __PRELOADED_REDUX_STATE__?: TPreloadedState;
}

declare const window: Window;

declare module '*.scss' {
    import classnames from 'classnames';

    export default classnames;
}
// eslint-disable-next-line @typescript-eslint/naming-convention
declare type NonNullableValues<T> = {[P in keyof T]: NonNullable<T[P]>};
// eslint-disable-next-line @typescript-eslint/naming-convention
declare type Nullable<T> = null | T;
// eslint-disable-next-line @typescript-eslint/naming-convention
declare type PartialRecord<K, V> = Partial<Record<K, V>>;
// eslint-disable-next-line @typescript-eslint/naming-convention
declare type WithNullValues<T> = {[P in keyof T]: T[P] | null};
// eslint-disable-next-line @typescript-eslint/naming-convention
declare type TypeOfPromise<T extends Promise<any>> = T extends Promise<infer R>
    ? R
    : never;

declare const __SERVER__: boolean;
declare const __CLIENT__: boolean;
declare const __DEV__: boolean;
declare const __PROD__: boolean;
declare const __REDUX_DEV_TOOLS__: boolean;
declare const __BUILD_LANG__: 'ru';

declare module '*.png' {
    const value: string;
    export default value;
}

declare module '*.svg' {
    const value: string;
    export default value;
}

declare module '*.ico' {
    const value: string;
    export default value;
}

declare module '*.bmp' {
    const value: string;
    export default value;
}

declare module '*.gif' {
    const value: string;
    export default value;
}

declare module '*.jpg' {
    const value: string;
    export default value;
}

declare module '*.jpeg' {
    const value: string;
    export default value;
}

declare module '*.webp' {
    const value: string;
    export default value;
}
