import {Dirent} from 'fs';

const {readdirSync} = require('fs');

function getFilesAndDirectories(
    path: string,
    excludeFolders: string[],
): {
    name: string;
    isDirectory: boolean;
    ext: string | null;
}[] {
    return (readdirSync(path, {withFileTypes: true}) as Dirent[])
        .map(dirent => ({
            name: dirent.name,
            isDirectory: dirent.isDirectory(),
            ext: dirent.isDirectory()
                ? null
                : dirent.name.split('.')[dirent.name.split('.').length - 1],
        }))
        .filter(
            direntInfo =>
                !direntInfo.isDirectory ||
                !excludeFolders.includes(direntInfo.name),
        );
}

function getConsoleType(tsFilesPercentage: number): string {
    if (tsFilesPercentage === 100) {
        return '\x1b[30m';
    } else if (tsFilesPercentage >= 80) {
        return '\x1b[32m';
    } else if (tsFilesPercentage >= 60) {
        return '\x1b[33m';
    }

    return '\x1b[31m';
}

// dev ветка, состояние на 13.08.2021
const prevFollowedFoldersStats = {jsFilesCount: 298, tsFilesCount: 5051};
const prevStats = JSON.parse(
    `{"./server/api":{"jsFilesCount":0,"tsFilesCount":433},"./server/configs":{"jsFilesCount":11,"tsFilesCount":0},"./server/constants":{"jsFilesCount":3,"tsFilesCount":4},"./server/controllers":{"jsFilesCount":10,"tsFilesCount":60},"./server/loggers":{"jsFilesCount":1,"tsFilesCount":57},"./server/middlewares":{"jsFilesCount":29,"tsFilesCount":0},"./server/providers":{"jsFilesCount":22,"tsFilesCount":3},"./server/redux":{"jsFilesCount":6,"tsFilesCount":62},"./server/render":{"jsFilesCount":0,"tsFilesCount":2},"./server/services":{"jsFilesCount":2,"tsFilesCount":165},"./server/utilities":{"jsFilesCount":28,"tsFilesCount":52},"./server":{"jsFilesCount":116,"tsFilesCount":839},"./integration/@types":{"jsFilesCount":0,"tsFilesCount":1},"./integration/client":{"jsFilesCount":0,"tsFilesCount":1},"./integration/commands":{"jsFilesCount":10,"tsFilesCount":2},"./integration/configuration":{"jsFilesCount":4,"tsFilesCount":0},"./integration/helpers":{"jsFilesCount":12,"tsFilesCount":364},"./integration/plugins":{"jsFilesCount":9,"tsFilesCount":0},"./integration/suites":{"jsFilesCount":1,"tsFilesCount":5},"./integration/testpalm-plugins":{"jsFilesCount":1,"tsFilesCount":0},"./integration/tests":{"jsFilesCount":0,"tsFilesCount":101},"./integration":{"jsFilesCount":42,"tsFilesCount":474},"./src/components":{"jsFilesCount":8,"tsFilesCount":681},"./src/constants":{"jsFilesCount":2,"tsFilesCount":54},"./src/containers":{"jsFilesCount":0,"tsFilesCount":31},"./src/contexts":{"jsFilesCount":0,"tsFilesCount":10},"./src/helpers":{"jsFilesCount":1,"tsFilesCount":0},"./src/hooks":{"jsFilesCount":0,"tsFilesCount":4},"./src/icons":{"jsFilesCount":6,"tsFilesCount":443},"./src/polyfills":{"jsFilesCount":0,"tsFilesCount":4},"./src/prerequisites":{"jsFilesCount":0,"tsFilesCount":291},"./src/projects":{"jsFilesCount":2,"tsFilesCount":2552},"./src/redux":{"jsFilesCount":35,"tsFilesCount":971},"./src/serviceProvider":{"jsFilesCount":0,"tsFilesCount":45},"./src/types":{"jsFilesCount":0,"tsFilesCount":178},"./src/utilities":{"jsFilesCount":10,"tsFilesCount":292},"./src":{"jsFilesCount":65,"tsFilesCount":5557},"./src/projects/account":{"jsFilesCount":2,"tsFilesCount":182},"./src/projects/avia":{"jsFilesCount":0,"tsFilesCount":621},"./src/projects/buses":{"jsFilesCount":0,"tsFilesCount":99},"./src/projects/favorites":{"jsFilesCount":0,"tsFilesCount":15},"./src/projects/happyPage":{"jsFilesCount":0,"tsFilesCount":31},"./src/projects/hotels":{"jsFilesCount":0,"tsFilesCount":473},"./src/projects/incompatible":{"jsFilesCount":0,"tsFilesCount":2},"./src/projects/index":{"jsFilesCount":0,"tsFilesCount":45},"./src/projects/partners":{"jsFilesCount":0,"tsFilesCount":25},"./src/projects/subscription":{"jsFilesCount":0,"tsFilesCount":2},"./src/projects/testControlPanel":{"jsFilesCount":0,"tsFilesCount":28},"./src/projects/trains":{"jsFilesCount":0,"tsFilesCount":1029}}`,
);

const accStats: Record<string, {jsFilesCount: number; tsFilesCount: number}> =
    {};

function analizeAll(
    path: string,
    excludeFolders: string[],
    depth: number,
): {jsFilesCount: number; tsFilesCount: number} {
    let jsFilesCount = 0;
    let tsFilesCount = 0;

    getFilesAndDirectories(path, excludeFolders).forEach(direntInfo => {
        if (direntInfo.isDirectory) {
            const {
                jsFilesCount: jsFilesCountInside,
                tsFilesCount: tsFilesCountInside,
            } = analizeAll(
                `${path}/${direntInfo.name}`,
                excludeFolders,
                depth + 1,
            );

            jsFilesCount += jsFilesCountInside;
            tsFilesCount += tsFilesCountInside;
        } else if (direntInfo.ext === 'js') {
            jsFilesCount++;
        } else if (direntInfo.ext === 'ts' || direntInfo.ext === 'tsx') {
            tsFilesCount++;
        }
    });

    const jsAndTsFilesCount = jsFilesCount + tsFilesCount;

    if (jsAndTsFilesCount !== 0 && depth <= 1) {
        const jsFilesPercentage = Number(
            ((jsFilesCount / jsAndTsFilesCount) * 100).toFixed(2),
        );
        const tsFilesPercentage = Number(
            ((tsFilesCount / jsAndTsFilesCount) * 100).toFixed(2),
        );

        console.log(
            getConsoleType(tsFilesPercentage),
            `${path} JS: ${jsFilesCount} files, ${jsFilesPercentage}%. TS: ${tsFilesCount} files, ${tsFilesPercentage}%.`,
        );

        accStats[path] = {
            jsFilesCount,
            tsFilesCount,
        };
    }

    return {
        jsFilesCount,
        tsFilesCount,
    };
}

const serverStats = analizeAll(
    './server',
    ['node_modules', 'build', 'html-report', '.idea', '.git', '_'],
    0,
);

console.log('\n');

const integrationStats = analizeAll(
    './integration',
    ['node_modules', 'build', 'html-report', '.idea', '.git', '_'],
    0,
);

console.log('\n');

const srcStats = analizeAll(
    './src',
    ['node_modules', 'build', 'html-report', '.idea', '.git', '_'],
    0,
);

console.log('\n');

analizeAll(
    './src/projects',
    ['node_modules', 'build', 'html-report', '.idea', '.git', '_'],
    0,
);

console.log('\x1b[30m', '\n');

const differentStats: string[] = [];

Object.entries(accStats).forEach(([path, {jsFilesCount, tsFilesCount}]) => {
    const allFilesCount = jsFilesCount + tsFilesCount;

    const {jsFilesCount: prevJsFilesCount, tsFilesCount: prevTsFilesCount} =
        prevStats[path] || {};

    const prevAllFilesCount = prevJsFilesCount + prevTsFilesCount;

    if (prevJsFilesCount !== undefined && prevTsFilesCount !== undefined) {
        if (Math.abs(prevJsFilesCount - jsFilesCount) > 5) {
            differentStats.push(
                `* ${path} js: ${prevJsFilesCount} (${(
                    (prevJsFilesCount / prevAllFilesCount) *
                    100
                ).toFixed(2)}%) -> ${jsFilesCount} (${(
                    (jsFilesCount / allFilesCount) *
                    100
                ).toFixed(2)}%)`,
            );
        }
    }
});

console.log(differentStats.join('\n'));

const followedFoldersStats = {
    jsFilesCount:
        serverStats.jsFilesCount +
        integrationStats.jsFilesCount +
        srcStats.jsFilesCount,
    tsFilesCount:
        serverStats.tsFilesCount +
        integrationStats.tsFilesCount +
        srcStats.tsFilesCount,
};

console.log(
    `\nОбщее по папкам (/server, /integration, /src): js: ${
        prevFollowedFoldersStats.jsFilesCount
    } (${(
        (prevFollowedFoldersStats.jsFilesCount /
            (prevFollowedFoldersStats.jsFilesCount +
                prevFollowedFoldersStats.tsFilesCount)) *
        100
    ).toFixed(2)}%) -> ${followedFoldersStats.jsFilesCount} (${(
        (followedFoldersStats.jsFilesCount /
            (followedFoldersStats.jsFilesCount +
                followedFoldersStats.tsFilesCount)) *
        100
    ).toFixed(2)}%).
`,
);

console.log('Положить в переменную prevStats:');
console.log(JSON.stringify(accStats));

console.log('\nПоложить в переменную prevFollowedFoldersStats:');
console.log(followedFoldersStats);
