import * as crypto from 'crypto';
import {upperFirst, camelCase} from 'lodash';
import {IFriggaOptions, EExportMode} from '@yandex-data-ui/frigga';

const ICONS_RE = /^Icons\s*\/\s*(?<size>\d+)\s*\/\s*(?<name>.*)$/;

function parseComponentName(componentName: string): {
    size: string;
    name: string;
} {
    const matchResult = ICONS_RE.exec(componentName);

    if (!matchResult?.groups) {
        throw new Error('Unexpected component name');
    }

    const {size, name} = matchResult.groups;

    return {size, name};
}

export const file: IFriggaOptions['file'] = 'I8EBpOVkBHKTHVrDyWuaaIhI';

export const select: IFriggaOptions['select'] = ({name}) => {
    return ICONS_RE.test(name);
};

export const getFileName: IFriggaOptions['getFileName'] = (
    component,
    exportMode,
) => {
    const extension = exportMode === EExportMode.SVG ? 'svg' : 'tsx';
    const {size, name} = parseComponentName(component.name);

    return `${size}/${upperFirst(camelCase(name))}.${extension}`;
};

export const getComponentName: IFriggaOptions['getComponentName'] =
    component => {
        const {name} = parseComponentName(component.name);

        return upperFirst(camelCase(name)) + 'Icon';
    };

export const getSvgrConfig: IFriggaOptions['getSvgrConfig'] = component => {
    const {size} = parseComponentName(component.name);

    const prefix = crypto
        .createHash('md5')
        .update(component.name)
        .digest('hex')
        .slice(-8);

    return {
        /**
         * В некоторых иконках {...props} затираем fill='none'
         * @link https://react-svgr.com/docs/options/#expand-props
         */
        expandProps: 'start',
        template: ({componentName, jsx}, {tpl}) => {
            return tpl`
          import {ReactElement} from 'react';
          ${'\n'}
          import {IIconProps} from 'icons/types/icon';
          ${'\n'}
          const ${componentName} = (props: IIconProps): ReactElement => ${jsx};
          ${'\n'}
          ${componentName}.defaultProps = {
            width: ${size.toString()},
            height: ${size.toString()},
            fill: 'currentColor',
          };
          ${'\n'}
          export default ${componentName};
        `;
        },
        typescript: true,
        dimensions: false,
        replaceAttrValues: {'#0D0D0D': '{props.fill}'},
        svgProps: {
            focusable: 'false',
        },
        svgoConfig: {
            plugins: [
                {
                    name: 'prefixIds',
                    params: {
                        prefix,
                    },
                },
            ],
        },
    };
};
