require('ts-node/register');
require('tsconfig-paths/register');

const utils = require('./configuration');
const config = require('./.e2e.json');

const {parameters, browsers, system} = utils;

const env = parameters.getEnvOnlyValue('E2E_ENV');
const envConfig = parameters.getStaticValue(config, env, {});

// parameters
const baseUrl = parameters.getEnvOnlyValue('E2E_URL');
const gridUrl = parameters.getValue(
    'E2E_GRID',
    envConfig,
    'grid',
    'http://localhost:4444/wd/hub',
);
const retries = parseInt(
    parameters.getValue('E2E_RETRIES', envConfig, 'retries', 10),
    10,
);
const reportPath = parameters.getValue(
    'E2E_REPORT_PATH',
    envConfig,
    'reportPath',
    'report',
);
const workersCount = parseInt(
    parameters.getValue('E2E_WORKERS_COUNT', envConfig, 'workersCount', 4),
    10,
);
const testsPerSession = parseInt(
    parameters.getValue(
        'E2E_TESTS_PER_SESSION',
        envConfig,
        'testsPerSession',
        3,
    ),
    10,
);
const sessionsPerBrowser = parseInt(
    parameters.getValue(
        'E2E_SESSIONS_PER_BROWSER',
        envConfig,
        'sessionsPerBrowser',
        10,
    ),
    10,
);
const isCI = Boolean(process.env.CI);

// authorization parameters
const passportEnv = parameters.getValue(
    'E2E_PASSPORT_ENV',
    envConfig,
    'passportEnv',
    'testing',
);

// constants
const WAIT_TIMEOUT = 2500;

const commonBrowsers = browsers.mapDefaultBrowserVersions(
    parameters.getStaticValue(
        envConfig,
        'sets.common.browsers',
        browsers.common,
    ),
);
const desktopBrowsers = browsers.mapDefaultBrowserVersions(
    parameters.getStaticValue(
        envConfig,
        'sets.desktop.browsers',
        browsers.desktop,
    ),
);
const touchBrowsers = browsers.mapDefaultBrowserVersions(
    parameters.getStaticValue(envConfig, 'sets.touch.browsers', browsers.touch),
);
const allBrowsers = [
    ...new Set([...commonBrowsers, ...desktopBrowsers, ...touchBrowsers]),
];

const testRetriesCounts = {};

const hermioneConfig = {
    sets: {
        common: {
            files: 'tests/common',
            browsers: commonBrowsers,
        },
        desktop: {
            files: 'tests/desktop',
            browsers: desktopBrowsers,
        },
        touch: {
            files: 'tests/touch',
            browsers: touchBrowsers,
        },
    },

    browsers: browsers.mapBrowserDescriptions({
        descriptions: allBrowsers,
        passportEnv,
        baseUrl,
        sessionsPerBrowser,
    }),

    retry: retries,
    waitTimeout: WAIT_TIMEOUT,
    testsPerSession,
    sessionRequestTimeout: 150000,

    // XXX Please define a flat map of primitive values, html-reporter rendering breaks on objects
    meta: {
        passportEnv,
        baseUrl,
    },

    system: system.getSystemConfiguration({workersCount}),

    plugins: {
        '@yandex-int/wdio-polyfill': {
            enabled: true,
            browsers: browsers.prepareFirefoxWebDriverPolyfills(
                env,
                allBrowsers,
            ),
        },

        'html-reporter/hermione': {
            enabled: true,
            path: `html-report/${reportPath}`,
            defaultView: 'all',
        },

        'json-reporter/hermione': {
            // E2E_ENABLE_JSON_REPORT - временный флаг для тимсити, убрать после полного переезда на CI
            enabled: isCI || process.env.E2E_ENABLE_JSON_REPORT === 'true',
            path: 'json-report/report.json',
        },

        'hermione-global-hook': {
            enabled: true,
            beforeEach: async function () {
                const {currentTest} = this;

                if (currentTest) {
                    const {browserId} = currentTest;
                    const testTitle = currentTest.fullTitle();

                    const testRetries = (testRetriesCounts[testTitle] =
                        testRetriesCounts[testTitle] || {});

                    if (!(browserId in testRetries)) {
                        testRetries[browserId] = -1;
                    }

                    this.retryCount = ++testRetries[browserId];
                } else {
                    this.retryCount = 0;
                }

                await this.browser.url('/');

                await this.browser.deleteCookies();

                await this.browser.setCookies({
                    name: 'e2e_test',
                    value: 'true',
                });

                try {
                    // browser.clearLocalStorage() не работает - https://github.com/webdriverio/webdriverio/issues/4068
                    await this.browser.execute(() => localStorage.clear());
                } catch (err) {
                    console.warn('Не получилось очистить localStorage');
                    console.warn(err);
                }
            },
        },

        [require.resolve('./plugins/createFallenTestsReport')]: {
            enabled:
                process.env.E2E_ENABLE_CREATE_FALLEN_TESTS_REPORT === 'true',
        },
        [require.resolve('./plugins/filterFallenTests')]: {
            enabled: process.env.E2E_ENABLE_FILTER_FALLEN_TESTS === 'true',
        },

        '@yandex-int/hermione-surfwax-router': {
            enabled: isCI,
        },

        '@yandex-int/hermione-muted-tests': {
            enabled: process.env.E2E_ENABLE_SKIP_TESTS_BY_TESTCOP === 'true',
            project: 'travel-frontend-portal',
            branch: 'trunk',
            stabilityIndex: {
                output: {
                    enabled: false,
                },
                input: {
                    path: null,
                },
            },
            testcopStats: {enabled: true},
            autoMute: {
                enabled: true,
            },
            sortTestsByStability: {
                enabled: false,
            },
        },
    },
    baseUrl,
    gridUrl,

    // Screenshots for assert view browser command
    screenshotsDir: ({file, browserId}) =>
        path.join(path.dirname(file), 'screenshots', browserId),

    prepareBrowser: function (browser) {
        /* Register common helpers */
        require('./commands/lib/login').default(browser);
        /* Register browserInfo helpers */
        require('./commands/lib/getBrowserName').default(browser);
        /* Register switch to tab helpers */
        require('./commands/lib/switchToTab').default(browser);
        /* Register clickJS helpers */
        require('./commands/lib/clickJS').default(browser);

        /* Set ru locale for moment */
        require('./helpers/utilities/date/setDateLocale').default();
    },
};

if (process.env.E2E_PARALLEL_LIMIT) {
    console.warn(
        'E2E_PARALLEL_LIMIT больше не используется, стоит использовать E2E_SESSIONS_PER_BROWSER',
    );
}

if (process.env.E2E_VERBOSE) {
    console.log(
        'e2e: hermione configuration:\n',
        JSON.stringify(hermioneConfig, null, 2),
    );
}

module.exports = hermioneConfig;
