/* Constants */

const QA_ATTRIBUTE_NAME = 'data-qa';
// TODO: если в production окружении атрибуты не нужны, то нужно определить это условие
const IS_PRODUCTION = false;
const EMPTY_QA_VALUE = undefined;
const EMPTY_QA_OBJECT = {};
const JOIN_QA_SYMBOL = '-';

/* Types */

export interface IWithQaAttributes {
    [QA_ATTRIBUTE_NAME]?: string;
}

export type TQaValue = string | undefined;
export type TQaProps = string | undefined | IObjectQAValue | IWithQaAttributes;

export interface IObjectQAValue {
    /**
     * Ключ элемента в списке
     */
    key?: string | number;
    /**
     * qa родителя
     */
    parent?: TQaProps;
    /**
     * qa текущего элемента
     */
    current?: TQaProps;
}

/* Utilities */

function instanceOfObjectQaValue(qaProps: TQaProps): qaProps is IObjectQAValue {
    return (
        typeof qaProps === 'object' &&
        ('key' in qaProps || 'parent' in qaProps || 'current' in qaProps)
    );
}

function instanceOfQaAttributes(
    qaProps: TQaProps,
): qaProps is IWithQaAttributes {
    return typeof qaProps === 'object' && QA_ATTRIBUTE_NAME in qaProps;
}

function instanceOfStringQaValue(qaProps: TQaProps): qaProps is string {
    return typeof qaProps === 'string';
}

function joinQaValues(qaValues: TQaValue[]): TQaValue {
    return qaValues.filter(Boolean).join(JOIN_QA_SYMBOL);
}

function getQa(qaProps: TQaProps): TQaValue {
    if (IS_PRODUCTION) {
        return EMPTY_QA_VALUE;
    }

    if (instanceOfStringQaValue(qaProps)) {
        return qaProps;
    }

    if (instanceOfQaAttributes(qaProps)) {
        return qaProps[QA_ATTRIBUTE_NAME];
    }

    if (instanceOfObjectQaValue(qaProps)) {
        const {key, parent, current} = qaProps;
        const qaKey = typeof key === 'undefined' ? undefined : String(key);
        const qaParent = getQa(parent);
        const qaCurrent = getQa(current);

        return joinQaValues([qaKey, qaParent, qaCurrent]);
    }

    return EMPTY_QA_VALUE;
}

export function prepareQaAttributes(qaProps: TQaProps): IWithQaAttributes {
    const qaValue = getQa(qaProps);

    if (IS_PRODUCTION || !qaValue) {
        return EMPTY_QA_OBJECT;
    }

    return {
        [QA_ATTRIBUTE_NAME]: qaValue,
    };
}
