import _isEqual from 'lodash/isEqual';

import {TFieldParams} from 'helpers/project/common/form/formFields';

import {Component} from 'components/Component';
import {ComponentArray} from 'components/ComponentArray';
import {Button} from 'components/Button';
import {Form} from 'components/Form';

import Document from './Document';

export default class Documents extends ComponentArray<Document> {
    readonly mobileAddButton: Button;
    readonly showAllButton: Button;
    readonly emptyText: Component;
    readonly form: Form;
    readonly formError: Component;
    readonly documentAddButtons: ComponentArray<Component>;

    constructor(
        browser: WebdriverIO.Browser,
        qa: QA = {current: 'item', parent: 'document'},
    ) {
        super(browser, qa, Document);

        this.mobileAddButton = new Button(browser, 'document-add-mobile');
        this.showAllButton = new Button(browser, 'document-add-more');

        this.emptyText = new Component(browser, 'documents-empty-text');
        this.documentAddButtons = new ComponentArray(
            browser,
            {parent: 'document', current: 'add-button'},
            Component,
        );

        this.form = new Form(
            browser,
            'document-edit-form',
            'document-edit-form-submit',
        );

        this.formError = new Component(browser, 'form-error-display');
    }

    async clickDocumentAddIntent(documentName: string): Promise<void> {
        const button = await this.documentAddButtons.findSeries(
            async element => {
                return (await element.getText()) === documentName;
            },
        );

        if (!button) {
            throw new Error(
                `Нет кнопки добавить для документа "${documentName}"`,
            );
        }

        await button.click();
    }

    async findDocumentsByFields(fields: {
        [key: string]: string | undefined;
    }): Promise<Document[]> {
        return await this.filter(async (document: Document) => {
            await document.waitForLoadingEnd();

            return _isEqual(await document.getData(), fields);
        });
    }

    async isSingleDocumentByFields(fields: {
        [key: string]: string | undefined;
    }): Promise<Document> {
        /*
         * Модал прячется за пределы экрана, webdriver не считает, что это скрытый элемент.
         * Поэтому тут проверяем, что форма не во вьюпорте.
         *
         * TODO: нужно использовать тестовый компонент модала, внутри которого уже будет компонент формы
         */
        await this.form.waitUntil(
            async () => (await this.form.isDisplayedInViewport()) === false,
        );

        const foundDocuments = await this.findDocumentsByFields(fields);

        if (!foundDocuments || foundDocuments.length === 0) {
            throw new Error('Не найдено документов с такими данными');
        } else if (foundDocuments && foundDocuments.length > 1) {
            throw new Error('Слишком много документов с такими данными');
        }

        return foundDocuments[0];
    }

    /**
     * Добавляем документ
     *
     * @param {{id: string, type: string, value: string}[]} fields
     * @param {string} documentName
     */

    async addDocument(
        fields: TFieldParams[],
        documentName: string,
    ): Promise<void> {
        await this.showAllButton.click();
        await this.clickDocumentAddIntent(documentName);

        await this.form.fill(fields);
        await this.form.submit();
    }

    /**
     * Редактируем документ
     * ВНИМАНИЕ: предварительно необходимо открыть форму редактирования нужного документа
     *
     * @param {{id: string, type: string, value: string}[]} newFields
     */

    async editDocument(newFields: TFieldParams[]): Promise<void> {
        await this.form.fill(newFields);
        await this.form.submit();
    }

    async removeAllDocuments(): Promise<void> {
        const documents = await this.items;

        for (let i = documents.length - 1; i >= 0; i--) {
            await documents[i].remove.click();
            await documents[i].removeConfirm.click();
        }

        await this.emptyText.waitForVisible();
    }
}

module.exports = Documents;
