/* eslint-disable camelcase */
import {SearchSuggest} from 'components/SearchSuggest';
import {DatePicker} from 'components/DatePicker';
import {Button} from 'components/Button';
import {Component} from 'components/Component';
import {SearchButtonGroup} from 'components/SearchButtonGroup';
import {AdFoxBanner} from 'components/AdFoxBanner';
import {TestModal} from 'components/TestModal';

import {AviaTravellers} from './AviaTravellers';

export interface IAviaSearchFormParams {
    adult_seats?: string | number;
    children_seats?: string | number;
    infant_seats?: string | number;
    fromName: string;
    klass?: 'business' | 'economy';
    return_date?: string;
    toName: string;
    when: string;
    fromPickFirst?: boolean;
    toPickFirst?: boolean;
}

export class AviaSearchForm extends Component {
    readonly fromSuggest: SearchSuggest;
    readonly toSuggest: SearchSuggest;

    readonly periodDatePicker: DatePicker;

    readonly travellers: AviaTravellers;

    readonly klass: SearchButtonGroup;

    readonly submitButton: Button;

    readonly testModal: TestModal;

    private _adFox: AdFoxBanner;

    constructor(browser: WebdriverIO.Browser) {
        super(browser, 'search-form');

        this.fromSuggest = new SearchSuggest(this.browser, {
            parent: this.qa,
            current: 'search-suggest-from',
        });
        this.toSuggest = new SearchSuggest(this.browser, {
            parent: this.qa,
            current: 'search-suggest-to',
        });

        this.periodDatePicker = new DatePicker(this.browser, {
            parent: this.qa,
            current: 'date-picker',
        });

        this.travellers = new AviaTravellers(this.browser);

        this.klass = new SearchButtonGroup(
            this.browser,
            'avia-travellers-search-button-group',
        );

        this.submitButton = new Button(this.browser, {
            parent: this.qa,
            current: 'submit',
        });

        this._adFox = new AdFoxBanner(this.browser);

        this.testModal = new TestModal(this.browser);
    }

    submitForm(): Promise<void> {
        return this.submitButton.click();
    }

    /**
     * Зполняет форму поиска
     * @param {Object} params
     * @returns {void}
     */
    async fill(params: IAviaSearchFormParams): Promise<void> {
        const {
            fromName,
            toName,
            when,
            return_date,
            adult_seats,
            children_seats,
            infant_seats,
            klass,
            fromPickFirst,
            toPickFirst,
        } = params;

        await this._adFox.disableEvents();

        if (fromName) {
            await this.fromSuggest.setSuggestValue(fromName, fromPickFirst);
        }

        if (toName) {
            await this.toSuggest.setSuggestValue(toName, toPickFirst);
        }

        if (when) {
            await this.periodDatePicker.selectStartDate(when);

            await this.scrollPageToTop();
        }

        if (return_date) {
            await this.periodDatePicker.selectEndDate(return_date);

            await this.scrollPageToTop();
        }

        if (adult_seats || children_seats || infant_seats || klass) {
            await this.travellers.click();

            if (adult_seats) {
                await this.travellers.adultsCount.setCount(adult_seats);
            }

            if (children_seats) {
                await this.travellers.childrenCount.setCount(children_seats);
            }

            if (infant_seats) {
                await this.travellers.infantCount.setCount(infant_seats);
            }

            if (klass) {
                await this.klass.clickByValue(klass);
            }

            if (this.isTouch) {
                await this.travellers.modal.clickCompleteButton();
            }
        }

        await this._adFox.enableEvents();
    }

    /**
     * Костыль: селениум при выборе даты внутри календаря видимо скроллит страницу до даты, чтобы по ней кликнуть
     * Из-за скролла скрывается вся форма и не получается выбрать следующие поля
     * Поэтому насильно скроллим страницу вверх после выбора даты
     *
     * Если при починке https://st.yandex-team.ru/TRAVELFRONT-7794 форма будет оставаться на месте при скролле
     * если одно из полей сфокуссировано, то костыль нужно будет убрать
     */
    async scrollPageToTop(): Promise<void> {
        await this.browser.execute(() => window.scrollTo(0, 0));

        /**
         * Ждём анимацию появления формы
         */
        await this.browser.pause(200);
    }
}
