import {
    CommandOptions,
    Value,
} from 'webdriverio/build/commands/element/addValue';

import {SECOND} from 'helpers/constants/dates';

import {ParsedCSSValue} from 'webdriverio/build/types';

import prepareQaSelector from 'commands/lib/utils/qa';

export interface IConstructable<T extends Component> {
    new (browser: WebdriverIO.Browser, qa: QA): T;
}

type TQaPart = string | number | undefined | null;

export class Component implements IComponent {
    qa: string = '';

    protected readonly browser: WebdriverIO.Browser;

    protected _parentPath: string[] | undefined;

    constructor(browser: WebdriverIO.Browser, qa?: QA) {
        this.browser = browser;

        if (typeof qa === 'string') {
            this.qa = qa;
        } else if (typeof qa === 'object') {
            if (typeof qa.parent === 'object') {
                this.qa = this.joinQa(qa.key, qa.current);
                this._parentPath = qa.parent.path;
            } else {
                this.qa = this.joinQa(qa.key, qa.parent, qa.current);
                this._parentPath = qa.path;
            }
        }
    }

    get selector(): string {
        return this.prepareQaSelector(this.qa);
    }

    get path(): string[] {
        return this.getPathWithQa(this.qa);
    }

    get isTouch(): boolean {
        const chromeOptions =
            this.browser.requestedCapabilities['goog:chromeOptions'] || {};

        /**
         * Хак: сейчас у нас все мобильные устройства - это мобильная эмуляция на базе хром.
         * Для остальных эмуляций или реальных мобильных девайсов
         * это условие придется доработать
         */
        return Boolean(chromeOptions.mobileEmulation);
    }

    get isDesktop(): boolean {
        return !this.isTouch;
    }

    async waitForVisible(timeout: number = 2000): Promise<true | void> {
        const element = this.browser.$(this.selector);

        return element.waitForDisplayed({
            timeout,
            timeoutMsg: `Element: "${this.selector}" not displayed after ${timeout}`,
        });
    }

    async waitForExist(timeout?: number): Promise<void | true> {
        return this.browser.$(this.selector).waitForExist({timeout});
    }

    async waitForHidden(timeout?: number): Promise<void | true> {
        const element = this.browser.$(this.selector);

        return element.waitForDisplayed({
            timeout,
            reverse: true,
            timeoutMsg: `Element: "${this.selector}" not hidden after ${timeout}`,
        });
    }

    async waitUntil(
        conditionFunction: () => Promise<boolean>,
        timeout: number = 2000,
        timeoutMsg?: string,
    ): Promise<void> {
        await this.browser.waitUntil(
            async () => {
                return await conditionFunction();
            },
            {
                timeout,
                timeoutMsg,
            },
        );
    }

    async getText(timeout?: number): Promise<string> {
        const element = this.browser.$(this.selector);

        await element.waitForExist({timeout});

        return element.getText();
    }

    async getHTML(timeout?: number): Promise<string | null> {
        const element = this.browser.$(this.selector);

        await element.waitForExist({timeout});

        return element.getHTML(false);
    }

    /** @depricated - устаревший метод, используйте isDisplayed, isDisplayedInViewport, waitForDisplayed */
    async isVisible(timeout = 3 * SECOND): Promise<boolean> {
        return this.isDisplayed(timeout);
    }

    async isDisplayed(timeout = 3 * SECOND): Promise<boolean> {
        try {
            await this.waitForVisible(timeout);

            return this.browser.$(this.selector).isDisplayed();
        } catch (e) {
            return false;
        }
    }

    async isExisting(timeout = 3 * SECOND): Promise<boolean> {
        try {
            await this.waitForExist(timeout);

            return this.browser.$(this.selector).isExisting();
        } catch (e) {
            return false;
        }
    }

    async isDisplayedInViewport(): Promise<boolean> {
        const element = await this.browser.$(this.selector);

        return await element.isDisplayedInViewport();
    }

    async getAttribute(attributeName: string): Promise<string | null> {
        const element = this.browser.$(this.selector);

        return element.getAttribute(attributeName);
    }

    async isDisabled(): Promise<boolean | null> {
        const element = this.browser.$(this.selector);

        const isEnabled = await element.isEnabled();

        return !isEnabled;
    }

    /**
     * Подскроливает к элементу, чтобы элемент оказался в центре viewport
     * @link https://webdriver.io/docs/api/element/scrollIntoView
     */
    async scrollIntoView(): Promise<void> {
        const element = await this.browser.$(this.selector);

        await element.scrollIntoView({block: 'center'});
    }

    async click(): Promise<void> {
        const element = this.browser.$(this.selector);

        await element.click();
    }

    /**
     * Клик по элементу через JS
     * Не рекомендуется, так как не эмулирует настоящий клик
     * Только в редких кейсах, например работа во фрейме
     */
    async clickJS(): Promise<void> {
        await this.browser.clickJS(this.selector);
    }

    async getCssProperty(propertyName: string): Promise<ParsedCSSValue> {
        return await this.browser.$(this.selector).getCSSProperty(propertyName);
    }

    async getValue(): Promise<string> {
        return this.browser.$(this.selector).getValue();
    }

    async setValue(
        value: Value | Value[],
        options?: CommandOptions,
    ): Promise<void> {
        return this.browser.$(this.selector).setValue(value, options);
    }

    async addValue(
        value: Value | Value[],
        options?: CommandOptions,
    ): Promise<void> {
        return this.browser.$(this.selector).addValue(value, options);
    }

    async clearValue(): Promise<void> {
        return this.browser.$(this.selector).clearValue();
    }

    protected prepareQaSelector(qa: string): string {
        if (this._parentPath) {
            return this.prepareQaSelectorWithPath(qa);
        }

        return prepareQaSelector(qa);
    }

    protected prepareQaSelectorWithPath(qa?: string): string {
        return this.getPathWithQa(qa).map(prepareQaSelector).join(' ');
    }

    protected getPathWithQa(qa?: string): string[] {
        const qaPart = qa ? [qa] : [];

        return (this._parentPath || []).concat(qaPart);
    }

    private joinQa(...parts: TQaPart[]): string {
        return parts.filter(key => key !== undefined && String(key)).join('-');
    }
}
