import moment, {Moment} from 'moment';

import dateFormats from 'helpers/utilities/date/formats';

import {SearchFormFieldModal} from './SearchFormFieldModal';
import {Calendar} from './Calendar';
import {DatePickerTrigger} from './DatePickerTrigger';
import {Component} from './Component';
import {retry} from '../retry';

export class DatePicker extends Component {
    startTrigger: DatePickerTrigger;
    endTrigger: DatePickerTrigger;

    modal: SearchFormFieldModal;
    calendar: Calendar;

    constructor(browser: WebdriverIO.Browser, qa: QA) {
        super(browser, qa);

        this.startTrigger = new DatePickerTrigger(browser, {
            parent: this.qa,
            current: 'start-trigger',
        });
        this.endTrigger = new DatePickerTrigger(browser, {
            parent: this.qa,
            current: 'end-trigger',
        });

        this.modal = new SearchFormFieldModal(this.browser);
        this.calendar = new Calendar(this.browser);
    }

    async selectStartDate(date: string): Promise<void> {
        const trigger = await this.startTrigger;

        await this.selectDate(trigger, date);
    }

    async selectEndDate(date: string): Promise<void> {
        const trigger = await this.endTrigger;

        await this.selectDate(trigger, date);
    }

    async getStartDate(): Promise<Moment> {
        return this.getDate(this.startTrigger);
    }

    async getEndDate(): Promise<Moment> {
        return this.getDate(this.startTrigger);
    }

    private async getDate(trigger: DatePickerTrigger): Promise<Moment> {
        const format = await this.getFormat();
        const dateString = await trigger.value.getText();

        return moment(dateString, format);
    }

    private async getFormat(): Promise<string> {
        if (await this.endTrigger.isVisible()) {
            return dateFormats.HUMAN_SHORT;
        }

        return dateFormats.HUMAN;
    }

    private async selectDate(
        trigger: DatePickerTrigger,
        date: string,
    ): Promise<void> {
        do {
            await trigger.click();
        } while (!(await this.calendar.isVisible()));

        await retry(async () => {
            //ждем анимацию
            await this.browser.pause(1000);

            await this.calendar.clickCalendarDate(date);
        })();
    }
}
