const program = require('commander');
const path = require('path');
const fse = require('fs-extra');
const chalk = require('chalk');

const USER = process.env.USER;
const DEFAULT_CONFIG_PATH = `/home/${USER}/nginx`;
const DEFAULT_CERTS_PATH = '/crt';
const DEFAULT_LOGS_PATH = `/home/${USER}/logs/ya-travel`;
const NGINX_CONFIG_PATH = '/etc/nginx/sites-enabled';
const DEFAULT_PORT = '9000';
const DEFAULT_CLIENT_PORT = '9001';
const CONFIG_NAME = 'ya-travel.conf';

const config = ({port, certsPath, logsPath, username}) => `
server {
    listen [::]:80;

    server_name ${username}.ui.yandex.ru ${username}-dev.sas.yp-c.yandex.net;

    return 301 https://$host$request_uri;
}

server {
    listen [::]:443 ssl;

    ssl_certificate ${certsPath}/dev.pem;
    ssl_certificate_key ${certsPath}/dev.pem;

    ssl_protocols TLSv1 TLSv1.1 TLSv1.2;
    ssl_ciphers AES128-SHA:AES256-SHA:RC4-SHA:DES-CBC3-SHA:RC4-MD5;
    ssl_session_cache shared:SSL:128m;
    ssl_session_timeout 28h;

    gzip on;
    gzip_types text/plain text/css application/javascript application/json application/xml;

    server_name travel.${username}.ui.yandex.ru travel.${username}-dev.sas.yp-c.yandex.net;

    access_log ${logsPath}/access.log;
    error_log ${logsPath}/error.log;

    set $project_path /home/${username}/ya-travel;

    location /webpackStatic/ {
        proxy_pass http://127.0.0.1:${DEFAULT_CLIENT_PORT}/webpackStatic/;
    }

    location /__webpack_hmr {
        proxy_pass http://127.0.0.1:${DEFAULT_CLIENT_PORT}/__webpack_hmr;
    }

    location / {
        try_files $uri @node;
    }

    location /favicon.ico {
        alias $project_path/build/public/favicons/favicon.ico;
    }

    location /static/ {
        alias $project_path/build/;
    }

    location ~ ^(/rooms/|/search/.*)$ {
        return 301 https://$host/hotels/;
    }

    location @node {
        proxy_pass http://127.0.0.1:${port};
        proxy_http_version 1.1;
        proxy_cache_bypass $http_upgrade;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_buffer_size 16k;
        proxy_buffers 8 16k;
        proxy_redirect off;
    }
}
`;

program.version('0.0.1').description('Setup nginx for development');

program
    .command('init')
    .description('Create nginx config for ya-travel project')
    .option(
        '-c, --config-path [configPath]',
        'Specify local config path',
        DEFAULT_CONFIG_PATH,
    )
    .option(
        '-l, --logs-path [logsPath]',
        'Specify logs path',
        DEFAULT_LOGS_PATH,
    )
    .option(
        '-ce, --certs-path [certsPath]',
        'Specify logs path',
        DEFAULT_CERTS_PATH,
    )
    .option('-p, --port [port]', 'Specify port', DEFAULT_PORT)
    .action(async ({port, configPath, certsPath, logsPath}) => {
        // Step 0: create log and nginx folders
        const logs = path.resolve(__dirname, logsPath);
        const configs = path.resolve(__dirname, configPath);

        console.log(
            chalk.green(
                `Creating nginx and logs folders at "${chalk.blue(
                    configs,
                )}" and "${chalk.blue(logs)}"...`,
            ),
        );

        try {
            await fse.ensureDir(logs);
            await fse.ensureDir(configs);
        } catch (ex) {
            console.error(chalk.red('Error occurred when creating folder'), ex);
            throw new Error(
                'ya-travel-scripts: Could not create nginx and logs folders',
            );
        }

        // Step 1: create config
        const configFilePath = path.resolve(__dirname, configPath, CONFIG_NAME);

        console.log(
            chalk.green(
                `Preparing config at "${chalk.blue(configFilePath)}"...`,
            ),
        );

        try {
            await fse.outputFile(
                configFilePath,
                config({port, certsPath, logsPath, username: USER}),
            );
        } catch (ex) {
            console.error(chalk.red('Error occured when preparing config'), ex);
            throw new Error('ya-travel-scripts: Could not prepare config');
        }

        const nginxConfigFilePath = path.resolve(
            __dirname,
            NGINX_CONFIG_PATH,
            CONFIG_NAME,
        );

        // Step 2: link config
        try {
            console.log(
                chalk.green(
                    `Linking config to "${chalk.blue(nginxConfigFilePath)}"...`,
                ),
            );
            await fse.ensureSymlink(configFilePath, nginxConfigFilePath);
        } catch (ex) {
            console.error(chalk.red('Error occured when linking config'), ex);
            throw new Error('ya-travel-scripts: Could not link config');
        }
    });

program.parse(process.argv);
