import {
    IConfirmSubscriptionAnswer,
    IConfirmSubscriptionsParams,
    ISubscribeByEmailAnswer,
    ISubscribeByEmailParams,
    ISubscribeParams,
    ISubscriptionsAnswer,
    ISubscriptionsParams,
    IUnsubscribeByDirectionParams,
    IUnsubscribeParams,
} from 'server/api/AviaApi/types/subscriptions';
import {EApiEntry} from 'types/EApiEntry';

import {RestApiClient} from 'server/utilities/RestApiClient';

import {IDependencies} from 'server/getContainerConfig';

export class AviaApi extends RestApiClient {
    private uid?: string;
    private yandexuid: string;

    constructor({
        logger,
        httpClient,
        requestId,
        getApiHost,
        getSrcParams,
        userInfo,
        yandexuid,
        sendClickHouseStats,
        rootSpan,
    }: IDependencies) {
        super({
            httpClient,
            baseURL: `${getApiHost(EApiEntry.AVIA)}`,
            logger,
            requestId,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.AVIA);
        this.uid = userInfo && 'uid' in userInfo ? userInfo.uid : undefined;
        this.yandexuid = yandexuid;
    }

    subscriptions(params: ISubscriptionsParams): Promise<ISubscriptionsAnswer> {
        return this.get<{data: ISubscriptionsAnswer}>(
            '/email_subscription/v1.0/user_subscriptions',
            {
                params: {
                    ...params,
                    passportuid: this.uid,
                },
            },
        ).then(({data}) => data);
    }

    subscribe(params: ISubscribeParams): Promise<void> {
        return this.post('/email_subscription/v1.0/subscribe', {
            qid: params.qid,
            // TODO: копнуть логику с платформами
            email_source: 'web',
            passportuid: this.uid,
            min_price: JSON.stringify(params.minPrice),
        });
    }

    subscribeByEmail(
        params: ISubscribeByEmailParams,
    ): Promise<ISubscribeByEmailAnswer['data']> {
        return this.post<ISubscribeByEmailAnswer>(
            '/email_subscription/v1.0/subscribe',
            {
                qid: params.qid,
                email_source: 'web',
                email: params.email,
                passportuid: this.uid || '',
                session: this.yandexuid,
                min_price: JSON.stringify(params.minPrice),
            },
        ).then(({data}) => data);
    }

    unsubscribeByDirection(
        params: IUnsubscribeByDirectionParams,
    ): Promise<void> {
        return this.post(
            '/email_subscription/v1.0/unsubscribe/by_direction',
            params,
        );
    }

    unsubscribe(params: IUnsubscribeParams): Promise<void> {
        return this.post('/email_subscription/v1.0/unsubscribe', params);
    }

    confirmSubscription(
        params: IConfirmSubscriptionsParams,
    ): Promise<IConfirmSubscriptionAnswer> {
        return this.post<{data: IConfirmSubscriptionAnswer}>(
            '/email_subscription/v1.0/confirm',
            params,
        ).then(({data}) => data);
    }
}
