import {CommonHeaders} from 'server/constants/headers';

import {IBookingVariantsResponse} from './types/IBookingVariantsResponse';
import {ICompositeOrderStateDTO} from './types/ICompositeOrderStateDTO';
import {IInitOrderPaymentForm} from './types/IInitOrderPaymentForm';
import {IOrderDTO} from './types/IOrderDTO';
import {IAviaCreateOrderApiRequest} from './types/ICreateOrderRequest';
import {IConfirmationCompleteApiRequest} from './types/IConfirmationCompleteApiRequest';
import {EApiEntry} from 'types/EApiEntry';
import IAviaGetOrderApiParams from 'server/api/AviaBookingApi/types/IAviaGetOrderApiParams';

import {RestApiClient} from '../../utilities/RestApiClient';
import {IRequestConfig} from '../../utilities/HttpClient/IHttpClient';

import {IDependencies} from '../../getContainerConfig';

export class AviaBookingVariantsApi extends RestApiClient {
    private readonly serviceTicket: string;

    constructor({
        logger,
        httpClient,
        requestId,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL:
                getApiHost(EApiEntry.TRAVEL) + '/avia_booking_flow/v1/variants',
            requestId,
            logger,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
    }

    variants(token: string): Promise<IBookingVariantsResponse> {
        return this.get<IBookingVariantsResponse>('', {params: {id: token}});
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.timeout = 25000;
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
        };
    }
}

export class AviaBookingOrdersApi extends RestApiClient {
    private readonly serviceTicket: string;
    private readonly mockTrips: boolean;

    constructor({
        logger,
        httpClient,
        requestId,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        rootSpan,
        mockTrips,
    }: IDependencies) {
        super({
            baseURL:
                getApiHost(EApiEntry.TRAVEL) + '/avia_booking_flow/v1/orders',
            requestId,
            logger,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
        this.mockTrips = mockTrips;
    }

    async getOrder(params: IAviaGetOrderApiParams): Promise<IOrderDTO> {
        const response = await this.get<IOrderDTO>('info', {
            params,
        });

        if (this.mockTrips) {
            response.tripId = '1';
        }

        return response;
    }

    createOrder(data: IAviaCreateOrderApiRequest): Promise<IOrderDTO> {
        return this.post<IOrderDTO>('', data);
    }

    getCompositeState(orderId: string): Promise<ICompositeOrderStateDTO> {
        return this.get<ICompositeOrderStateDTO>('state', {
            params: {id: orderId},
        });
    }

    confirmationComplete(data: IConfirmationCompleteApiRequest): Promise<void> {
        return this.post<void>('confirmation_complete', data);
    }

    initPaymentAsync(
        orderId: string,
        initOrderPaymentForm: IInitOrderPaymentForm,
    ): Promise<void> {
        return this.post<void>('payment', initOrderPaymentForm, {
            params: {id: orderId},
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.timeout = 25000;
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
        };
    }
}
