import {IAviaBackendApiClient} from 'server/utilities/AviaBackendApiClient/IAviaBackendApiClient';
import {ILogger} from 'server/utilities/Logger';

import {IDependencies} from 'server/getContainerConfig';

export interface INationalClickPriceMap {
    ru: number;
    com: number;
    tr: number;
    ua: number;
    kz: number;
    [key: string]: number;
}

export interface IPartner {
    billingDatasourceId: number | null;
    billingOrderId: number;
    billingClientId: number;
    canSaleByInstallments: boolean;
    id: number;
    clickPrice: number;
    clickPriceNational: INationalClickPriceMap;
    eClickPriceNational: INationalClickPriceMap;
    code: string;
    enabled: boolean;
    marker: string | null;
    enabledInWizard: Record<string, boolean>;
}

export interface IPopularPartner {
    score: number;
    // eslint-disable-next-line camelcase
    partner_code: string;
}

export interface IPartnerInfo {
    logo: string | null;
    logoSvg: string;
    title: string;
}

export type TAviaPartnersMap = Map<string, IPartner>;

export class AviaPartnersApi {
    private api: IAviaBackendApiClient;
    private restApi: IAviaBackendApiClient;
    private logger: ILogger;
    private readonly nationalVersion: string;
    private lang = 'ru';

    constructor({
        nationalVersion,
        aviaGatewayBackendApiClient,
        aviaBackendRestApiClient,
        logger,
    }: IDependencies) {
        this.nationalVersion = nationalVersion;
        this.api = aviaGatewayBackendApiClient;
        this.restApi = aviaBackendRestApiClient;
        this.logger = logger;
    }

    getPopularPartners(
        fromPointKey: string,
        toPointKey: string,
    ): Promise<IPopularPartner[] | null> {
        const methodPath = 'partners/popular_by_route';
        const natPath = `${this.nationalVersion}/${this.lang}`;
        const geoPath = `${fromPointKey}/${toPointKey}`;

        return this.restApi.request<IPopularPartner[]>({
            name: `${methodPath}/${natPath}/${geoPath}`,
        });
    }

    getAll(): Promise<Map<string, IPartner>> {
        return this.api
            .request<IPartner[]>({
                name: 'partnersData',
            })
            .then(partners =>
                (partners || []).reduce(
                    (partnersByCode: Map<string, IPartner>, partner) => {
                        const key = this.getKey(partner.code);

                        partnersByCode.set(key, partner);

                        return partnersByCode;
                    },
                    new Map(),
                ),
            )
            .catch(e => {
                this.logger.logError(e);

                return new Map();
            });
    }

    getByCode(partnerCode: string): Promise<IPartner | null> {
        const key = this.getKey(partnerCode);

        return this.getAll()
            .then(partnersByCode => {
                return partnersByCode.get(key) || null;
            })
            .catch(e => {
                this.logger.logError(e);

                return null;
            });
    }

    getPartner(partnerCode: string): Promise<IPartnerInfo | null> {
        return this.api
            .request<IPartnerInfo>({
                name: 'partner',
                params: {
                    code: partnerCode,
                },
                fields: ['logo', 'logoSvg', 'title'],
            })
            .catch(e => {
                this.logger.logError(e);

                return null;
            });
    }

    private getKey(partnerCode: string): string {
        if (partnerCode.startsWith('dohop')) {
            partnerCode = 'dohop';
        }

        if (partnerCode.startsWith('amadeus')) {
            partnerCode = 'amadeus';
        }

        return partnerCode;
    }
}
