import snakeCaseKeys from 'snakecase-keys';

import {IAviaPriceIndexSearchDTO} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexSearchDTO';
import {IAviaPriceIndexSearchInterval} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexSearchInterval';
import {IAviaPriceIndexSearchParams} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexSearchParams';
import {TAviaPriceIndexFilters} from 'server/api/AviaPriceIndexApi/types/IAviaPriceIndexFilters';
import {EApiEntry} from 'types/EApiEntry';
import IAviaMinPriceBatchSearchApiParams from 'server/api/AviaPriceIndexApi/types/IAviaMinPriceBatchSearchApiParams';
import IAviaMinPriceBatchSearchApiResponse from 'server/api/AviaPriceIndexApi/types/IAviaMinPriceBatchSearchApiResponse';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from 'server/getContainerConfig';

export class AviaPriceIndexApi extends RestApiClient {
    private readonly nationalVersion: string;

    constructor({
        logger,
        httpClient,
        requestId,
        nationalVersion,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL: getApiHost(EApiEntry.AVIA_PRICE_INDEX),
            requestId,
            logger,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.AVIA_PRICE_INDEX);
        this.nationalVersion = nationalVersion;
        logger.addExtra({api: 'AviaPriceIndexApi'});
    }

    // https://a.yandex-team.ru/arcadia/travel/avia/price_index/views/search.md
    search(
        interval: IAviaPriceIndexSearchInterval,
        searchParams: IAviaPriceIndexSearchParams,
        filters?: TAviaPriceIndexFilters,
    ): Promise<IAviaPriceIndexSearchDTO> {
        const path = this.createSearchPath(interval, searchParams);

        return this.post<IAviaPriceIndexSearchDTO>(
            `search_methods/v1/search/${path}`,
            {
                filters,
            },
        );
    }

    minPriceBatchSearch(
        params: IAviaMinPriceBatchSearchApiParams,
    ): Promise<IAviaMinPriceBatchSearchApiResponse> {
        // @ts-ignore какие-то проблемы в типах snakeCaseKeys
        const snakeCaseParams = snakeCaseKeys(params, {deep: true});

        return this.post<IAviaMinPriceBatchSearchApiResponse>(
            `search_methods/v1/min_price_batch_search/${this.nationalVersion}`,
            snakeCaseParams,
            {convertResponseKeysToCamelCase: true},
        );
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.timeout = 25000;
    }

    private createSearchPath(
        interval: IAviaPriceIndexSearchInterval,
        searchParams: IAviaPriceIndexSearchParams,
    ): string {
        return [
            this.nationalVersion,
            interval.startDate,
            interval.endDate,
            searchParams.forwardDate,
            searchParams.backwardDate,
            searchParams.fromId,
            searchParams.toId,
            searchParams.adultsCount,
            searchParams.childrenCount,
            searchParams.infantsCount,
        ]
            .filter(value => value !== null)
            .join('/');
    }
}
