import {get} from 'lodash';

import {
    TAviaPriceIndexFilters,
    IAviaPriceIndexTransferFilter,
    IAviaPriceIndexTimeFilter,
    IAviaPriceIndexAirportFilter,
} from '../types/IAviaPriceIndexFilters';
import {EPriceIndexTimeFilter} from '../types/EPriceIndexTimeFilter';

import {
    IAviaSearchResultsFilters,
    EAviaSearchTransferFilter,
    EAviaSearchTimeFilter,
} from 'reducers/avia/search/results/filters/reducer';

import {isEnabled, toArray, Flag} from 'utilities/flags/flags';

const frontTimeFilterToPriceIndexFilter = {
    [EAviaSearchTimeFilter.MORNING]: EPriceIndexTimeFilter.MORNING,
    [EAviaSearchTimeFilter.DAY]: EPriceIndexTimeFilter.DAY,
    [EAviaSearchTimeFilter.EVENING]: EPriceIndexTimeFilter.EVENING,
    [EAviaSearchTimeFilter.NIGHT]: EPriceIndexTimeFilter.NIGHT,
};

// Сейчас в динамике используются только фильтры багажа, пересадок и времени, остальные просто игнорируем
export function convertSearchFiltersToPriceIndexFilters(
    searchFilters: Partial<IAviaSearchResultsFilters>,
): TAviaPriceIndexFilters {
    return {
        time: getTimeFilter(searchFilters),
        transfer: getTransferFilter(searchFilters),
        withBaggage: getBaggageFilter(searchFilters),
        airlines: getAirlinesFilter(searchFilters),
        airport: getAirports(searchFilters),
    };
}

function getAirlinesFilter(
    searchFilters: Partial<IAviaSearchResultsFilters>,
): TAviaPriceIndexFilters['airlines'] {
    return searchFilters.company?.companiesIds;
}

function getAirports(
    searchFilters: Partial<IAviaSearchResultsFilters>,
): IAviaPriceIndexAirportFilter {
    if (!searchFilters.airports) {
        return {
            forwardArrival: [],
            forwardDeparture: [],
            forwardTransfers: [],
            backwardArrival: [],
            backwardDeparture: [],
            backwardTransfers: [],
        };
    }

    const [forward, backward] = searchFilters.airports;

    return {
        forwardArrival: forward.arrival,
        forwardDeparture: forward.departure,
        forwardTransfers: forward.transfers,
        backwardArrival: backward.arrival,
        backwardDeparture: backward.departure,
        backwardTransfers: backward.transfers,
    };
}

function getTimeFilter(
    searchFilters: Partial<IAviaSearchResultsFilters>,
): IAviaPriceIndexTimeFilter {
    const forwardArrival = get(searchFilters, ['time', 0, 'arrival']);
    const forwardDeparture = get(searchFilters, ['time', 0, 'departure']);
    const backwardArrival = get(searchFilters, ['time', 1, 'arrival']);
    const backwardDeparture = get(searchFilters, ['time', 1, 'departure']);

    return {
        forwardArrival: getTimeFilterValue(forwardArrival),
        forwardDeparture: getTimeFilterValue(forwardDeparture),
        backwardArrival: getTimeFilterValue(backwardArrival),
        backwardDeparture: getTimeFilterValue(backwardDeparture),
    };
}

function getTimeFilterValue(value: Flag<number>): number[] | null {
    const filter = toArray(value)
        .map(f => frontTimeFilterToPriceIndexFilter[f])
        .filter(Boolean);

    return filter.length ? filter : null;
}

function getTransferFilter(
    searchFilters: Partial<IAviaSearchResultsFilters>,
): IAviaPriceIndexTransferFilter {
    const transferFilter: IAviaPriceIndexTransferFilter = {
        count: null,
        hasAirportChange: null,
        hasNight: null,
    };

    if (
        isEnabled(
            get(searchFilters, ['transfer', 'value']),
            EAviaSearchTransferFilter.NO_AIRPORT_CHANGE,
        )
    ) {
        transferFilter.hasAirportChange = false;
    }

    if (
        isEnabled(
            get(searchFilters, ['transfer', 'value']),
            EAviaSearchTransferFilter.EXCLUDE_NIGHTLY,
        )
    ) {
        transferFilter.hasNight = false;
    }

    if (
        isEnabled(
            get(searchFilters, ['transfer', 'value']),
            EAviaSearchTransferFilter.ONE_CHANGE,
        )
    ) {
        transferFilter.count = 1;
    }

    if (
        isEnabled(
            get(searchFilters, ['transfer', 'value']),
            EAviaSearchTransferFilter.NO_TRANSFERS,
        )
    ) {
        transferFilter.count = 0;
    }

    return transferFilter;
}

/*
PI умеет фильтровать цены по отсутствию багажа, поэтому здесь важно заменять
false на null, т.к. на фронте у нас нет сценария фильтрации "без багажа", но
непримененный фильтр с багажом имеет значение false
*/
function getBaggageFilter(
    searchFilters: Partial<IAviaSearchResultsFilters>,
): true | null {
    return get(searchFilters, ['baggage', 'enabled']) || null;
}
