import _ from 'lodash';
import {AxiosError} from 'axios';

import {IAviaTDApiSearchQuery} from './types/IAviaTDApiSearchQuery';
import {IAviaTDInitAnswer} from './types/IAviaTDInitAnswer';
import {IAviaTDAnswer} from './types/IAviaTDAnswer';
import {IAviaTDRedirectAnswer} from './types/IAviaTDRedirect';
import {IAviaTDApiOrderQuery} from './types/IAviaTDApiOrderQuery';
import {IAviaTDResultsQuery} from './types/IAviaTDResultsQuery';
import {IAviaTDRedirectQuery} from './types/IAviaTDRedirectQuery';
import {IUserSplit} from 'server/providers/experiments/types';
import {EApiEntry} from 'types/EApiEntry';
import {IUATraits} from 'types/common/IUATraits';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';
import {getAviaServiceName} from 'server/loggers/avia/utils/getAviaServiceName';

import {IDependencies} from 'server/getContainerConfig';

export class AviaTicketDaemonApi extends RestApiClient {
    private yandexuid: string;
    private username?: string;
    private isStaff?: boolean = false;
    private isTravelApp: boolean;
    private uid?: string;
    private login?: string | null;
    private readonly nationalVersion: string;
    private uatraits: IUATraits;
    private getUserSplit: () => Promise<IUserSplit>;

    constructor({
        httpClient,
        getApiHost,
        getSrcParams,
        logger,
        requestId,
        nationalVersion,
        sendClickHouseStats,
        userInfo,
        yandexuid,
        uatraits,
        getUserSplit,
        rootSpan,
        isTravelApp,
    }: IDependencies) {
        super({
            httpClient,
            baseURL: `${getApiHost(EApiEntry.AVIA_TICKET_DAEMON)}/jsendapi`,
            logger,
            requestId,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.AVIA_TICKET_DAEMON);
        this.yandexuid = yandexuid;
        this.getUserSplit = getUserSplit;
        this.nationalVersion = nationalVersion;
        this.isTravelApp = isTravelApp;
        this.uatraits = uatraits;

        if (userInfo && 'uid' in userInfo) {
            this.uid = userInfo.uid;
            this.login = userInfo.login;
            this.isStaff = userInfo.isStaff;
            this.username = userInfo.displayName;
        }
    }

    initByQuery(
        searchQuery: IAviaTDApiSearchQuery,
        ignoreOutdated?: boolean,
    ): Promise<IAviaTDInitAnswer> {
        return this.get<{data: IAviaTDInitAnswer}>(
            'front/init_search_by_query/',
            {
                params: {...searchQuery, ignoreOutdated},
                axiosRetry: {
                    retryCondition: (error: AxiosError): boolean =>
                        error.code === String(500),
                    retries: 2,
                },
            },
        ).then(x => x.data);
    }

    getOrder(params: IAviaTDApiOrderQuery): Promise<IAviaTDAnswer> {
        return this.get<{data: IAviaTDAnswer}>('v3/order', {
            params,
            axiosRetry: {
                retries: 2,
            },
        }).then(x => x.data);
    }

    redirect(
        params: IAviaTDRedirectQuery,
    ): Promise<{data: {data: IAviaTDRedirectAnswer}; status: number}> {
        return this.get<{data: {data: IAviaTDRedirectAnswer}; status: number}>(
            'v3/redirect',
            {
                params,
                withStatus: true,
                // 400 и 404 - некорректные параметры и не найдено предложений, прекращаем ретраи
                validateStatus: statusCode =>
                    statusCode <= 400 || statusCode === 404,
            },
        );
    }

    results({
        qid,
        cont,
        allowPortional = '1',
        test_context,
    }: IAviaTDResultsQuery): Promise<IAviaTDAnswer> {
        return this.get<{data: {data: IAviaTDAnswer}; status: number}>(
            `v3/results/${qid}/${cont}/${allowPortional}`,
            {
                params: {test_context},
                axiosRetry: {
                    retries: 2,
                },
                withStatus: true,
                // см коды ответов EAviaTDAnswerStatusCode
                validateStatus: code => code <= 400 || code === 500,
            },
        ).then(x => ({...x.data.data, status: x.status}));
    }

    protected async interceptRequest(config: IRequestConfig): Promise<void> {
        const {backFlags} = await this.getUserSplit();
        const query = config.params || {};

        config.params = this.getQuery(query, backFlags);
    }

    private getQuery(
        query: Record<string, any>,
        backFlags?: string,
    ): Record<string, string | boolean | undefined> {
        const clearedQuery = _.omitBy(query, _.isUndefined);

        return {
            lang: 'ru',
            // 'client_ip_country_geo_id': req.clientIpCountryGeoId,
            national: this.nationalVersion,
            // 'platform': req.platform,
            service: getAviaServiceName(this.uatraits, this.isTravelApp),

            user_info: JSON.stringify({
                yandexuid: this.yandexuid,
                passportuid: this.uid,
                yauser: {
                    uid: this.uid,
                    login: this.login,
                },
                django_user: {
                    username: this.username,
                    is_staff: this.isStaff,
                },
            }),
            back_flags: backFlags,
            ignoreOutdated: false,
            ...clearedQuery,
        };
    }
}
