import IPrice from 'utilities/currency/PriceInterface';

import {TVariantFareFamilyKeys, TFareFamilies} from './IAviaTDFareFamily';

/* eslint-disable camelcase */

export type TBaggageKey = string;

export interface IVariantPrice {
    /**
     * Список ключей багажных тарифов для каждого перелета
     * @see {IReference} инфорамация по багажным тарифам в справочнике baggageTariffs
     */
    baggage: TBaggageKey[][];
    /**
     * Ключ партнера
     * @see {IReference} инфорамация по партнерам в справочнике partners
     */
    partnerCode: string;
    /** Время ответа партнёра в секундах */
    queryTime: number;
    tariff: IPrice;
    /** Для логов */
    tariff_sign: string;
    /** Вариант продаёт авиакомпания */
    fromCompany?: boolean;
    charter?: boolean;
    service: string;
    fare_codes?: string[][][];
    /**
     * FareFamilies перелетов
     * @see {IReference} инфорамация по конкретному fareFamily находится в справочнике fareFamilies (иногда может отсутствовать)
     */
    fare_families?: TVariantFareFamilyKeys;
    fare_families_hash: string;
    boy: boolean;
    selfconnect: boolean;
    /** Бонусы Яндекс.Плюса */
    plusPoints?: number;
}

export type TRoute = TFlightKey[];

/** Тариф */
export interface IFare {
    hybrid?: boolean;
    /** Список цен от различных партнеров */
    prices: IVariantPrice[];
    /**
     * Список ключей перелётов
     * @see {IReference} инфорамация по перелетам находится в справочнике flights
     */
    route: [forward: TRoute, backward: TRoute];
}

export interface IVariants {
    /** Список тарифов */
    fares: IFare[];
}

export interface IAlliance {
    id: number;
    title: string;
}

export type TCostType = 'normal' | 'lowcost' | 'hybrid';

/**
 * Признак сокращенной ручной клади
 * @description Сокращенная ручная кладь - нет ни одного измерения больше 49
 * @example
 *  Ютейр 40х30х20 5
 *  Смартавиа 40х30х20 10
 *  Победа 36х30х27
 */
export enum ECarryOnType {
    /** сокращенная ручная кладь */
    SMALL = 'small',
    /** стандартная ручная кладь */
    REGULAR = 'regular',
}

export interface IAviaCompany {
    id: number;
    /** Признак сокращенной ручной клади */
    carryonSizeBucket: ECarryOnType;
    baggageDimensionsSum?: number;
    baggageRules?: string;
    baggageRulesUrl?: string;
    /** Высота ручной клади */
    carryonHeight?: number;
    /** Длина ручной клади */
    carryonLength?: number;
    /** Ширина ручной клади */
    carryonWidth?: number;
    costType: TCostType;
}

export interface IBaggageOption {
    count?: number | null;
    /** Откуда получена информация о багаже: от партнёра или из базы */
    source: 'partner' | 'db' | 'fare_families' | null;
    [key: string]: number | undefined | string | null;
}

export interface IBaggageTDTariff {
    included?: IBaggageOption;
    /** Количество сумок */
    pc?: IBaggageOption;
    /** Вес сумки в кг */
    wt?: IBaggageOption;
    [key: string]: IBaggageOption | undefined;
}

type TCompanyId = number;

export interface ICompany {
    id: TCompanyId;
    logoSvg: string;
    title: string;
    url: string;
    alliance: number | null;
    color?: string;
}

type TCompanyTariffId = number;

export interface ICompanyTariff {
    id: TCompanyTariffId;
    baggageAllowed: boolean;
    baggageNorm: number | null;
    carryon: boolean;
    carryonNorm: number | null;
    published: boolean;
}

export interface IFlightDate {
    local: string;
    tzname?: string;
    offset: number;
}

export type TFlightNumber = string;

export interface IFlightOperating {
    number: TFlightNumber;
    company: number;
}

export type TFlightKey = string;

/** Перелет */
export interface IFlight {
    key: TFlightKey;
    arrival: IFlightDate | null;
    /**
     * Код авиакомпании
     * @see {IReference} инфорамация по авиакомпаниям находится в справочнике aviaCompanies
     */
    aviaCompany?: number;
    /**
     * Код компании
     * @see {IReference} инфорамация по компаниям находится в справочнике companies
     */
    company: TCompanyId;
    /**
     * Тариф компании
     * @see {IReference} инфорамация по тарифам компаний находится в справочнике companyTariffs
     */
    companyTariff?: TCompanyTariffId;
    departure: IFlightDate | null;

    /**
     * От куда перелет
     * @see {IReference} справочник stations
     */
    from: TStationId;
    /**
     * Куда перелет
     * @see {IReference} справочник stations
     */
    to: TStationId;

    number: TFlightNumber;
    operating: IFlightOperating | null;
    tModel: string | null;
}

export interface ITDPartner {
    id: number;
    code: string;
    siteUrl?: string;
    logoSvg?: string | null;
    title: string;
}

export interface IRating {
    canceled?: number;
    /** Вероятности задержки от 0 до 100 */
    delayed3060?: number;
    delayed6090?: number;
    delayedLess30?: number;
    delayedMore90?: number;
    number: TFlightNumber;
    scores?: number;
}

/**
 * ID населенного пункта. Только не понятно, это TGeoId или TRaspId.
 * @example 75
 */
export type TSettlementId = number;
/** Населенный пункт */
export interface ISettlement {
    id: TSettlementId;
    /**
     * @example 'VVO'
     */
    code: string;
    /**
     * @example 210
     */
    countryId: number;
    /**
     * @example 'Владивостока'
     */
    phraseFrom?: string;
    /**
     * @example 'Владивостоке'
     */
    phraseIn?: string;
    /**
     * @example 'Владивосток'
     */
    phraseTo?: string;
    /**
     * @example 'Владивосток'
     */
    title: string;
    /**
     * Предлог (в/во/на) для формирования phraseIn/phraseTo
     * @example 'во'
     */
    preposition?: string;
}

/** Станция (аэропорт) */
export interface IStationType {
    prefix: string;
    title: string;
}

export type TStationType = 'plane' | 'bus' | 'train';

export type TStationId = number;

export interface IStation {
    id: TStationId;
    code: string;
    phraseFrom: string;
    phraseIn: string;
    phraseTo: string;
    /**
     * Населенный пункт
     * @see {IReference} справочник settlements
     */
    settlement: TSettlementId;
    stationType: IStationType;
    /** Тип станции */
    tType: TStationType;
    title: string;
    /** Предлог (в/во/на) для формирования phraseIn/phraseTo */
    preposition?: string;
}

/** Справочники */
export interface IReference {
    alliances: IAlliance[];
    /** Авиа компании */
    aviaCompanies: IAviaCompany[];
    /** Справочник багажных тарифов */
    baggageTariffs: Record<string, IBaggageTDTariff>;
    /** Компании */
    companies: ICompany[];
    /** Тарифы компаний */
    companyTariffs: ICompanyTariff[];
    /** Справочник перелетов */
    flights: IFlight[];
    /** Партнеры */
    partners: ITDPartner[];
    ratings: IRating[];
    reviewsCount: Record<string, number>;
    /** Населенные пункты */
    settlements: ISettlement[];
    /** Станции (аэропорты) */
    stations: IStation[];
    /** Семества тарифов */
    fareFamilies: TFareFamilies;
}

/** Статус опроса партнера */
export type TStatus = 'done' | 'querying' | 'outdated';

export enum EAviaTDAnswerStatusCode {
    /** Получили валидный ответ */
    OK = 200,
    /** Валидный ответ, но вариантов нет */
    NO_OFFERS = 204,
    /** Продолжает процесс опроса */
    POLL_CONTINUE = 206,
    /** Что-то не так с запросом, получили описание ошибки (останавливает ретраи) */
    WRONG_REQUEST = 400,
    /** Произошла внутренняя ошибка (ретрай 2 3 раза до 10 секунд, выставляем задержки между попытками 1 и 2 секунды, заканчиваем опрос и рисуем что есть) */
    INTERNAL_ERROR = 500,
}

/** Avia TicketDaemon Answer */
export interface IAviaTDAnswer {
    cont: number | null;
    /** Прогресс опроса партнеров */
    progress: IProgress;
    /** Справочник данных */
    reference: IReference;
    /** Список найденных вариантов */
    variants: IVariants;
    /** Статусы опросов партнеров (ключ - имя партнера, значение - статус) */
    partners?: Record<string, TStatus>;
    /** Идентификатор запроса */
    qid?: string;
    /** Код ответа бэка */
    status?: EAviaTDAnswerStatusCode;
}

export interface IProgress {
    /** Всего */
    all: number;
    /** Текущий прогресс */
    current: number;
}
