import {CommonHeaders} from 'server/constants/headers';

import IGenericOrderCreateOrderApiParams from 'server/api/GenericOrderApi/types/createOrder/IGenericOrderCreateOrderApiParams';
import IGenericOrderCreateOrderApiResponse from 'server/api/GenericOrderApi/types/createOrder/IGenericOrderCreateOrderApiResponse';
import IGenericOrderCancelOrderApiParams from 'server/api/GenericOrderApi/types/cancelOrder/IGenericOrderCancelOrderApiParams';
import IGenericOrderStartPaymentApiParams from 'server/api/GenericOrderApi/types/startPayment/IGenericOrderStartPaymentApiParams';
import IGenericOrderEstimateDiscountApiParams from 'server/api/GenericOrderApi/types/estimateDiscount/IGenericOrderEstimateDiscountApiParams';
import IGenericOrderEstimateDiscountApiResponse from 'server/api/GenericOrderApi/types/estimateDiscount/IGenericOrderEstimateDiscountApiResponse';
import IGenericOrderAddServiceApiParams from 'server/api/GenericOrderApi/types/addService/IGenericOrderAddServiceApiParams';
import IGenericOrderAddServiceApiResponse from 'server/api/GenericOrderApi/types/addService/IGenericOrderAddServiceApiResponse';
import IGenericOrderGetOrderStateApiParams from 'server/api/GenericOrderApi/types/getOrderState/IGenericOrderGetOrderStateApiParams';
import IGenericOrderGetOrderStateApiResponse from 'server/api/GenericOrderApi/types/getOrderState/IGenericOrderGetOrderStateApiResponse';
import IGenericOrderGetOrderApiParams from 'server/api/GenericOrderApi/types/getOrder/IGenericOrderGetOrderApiParams';
import IGenericOrderCalculateRefundAmountApiParams from 'server/api/GenericOrderApi/types/calculateRefundAmount/IGenericOrderCalculateRefundAmountApiParams';
import IGenericOrderCalculateRefundAmountApiResponse from 'server/api/GenericOrderApi/types/calculateRefundAmount/IGenericOrderCalculateRefundAmountApiResponse';
import IGenericOrderStartRefundApiParams from 'server/api/GenericOrderApi/types/startRefund/IGenericOrderStartRefundApiParams';
import IGenericOrderGetOrderApiResponse from 'server/api/GenericOrderApi/types/getOrder/IGenericOrderGetOrderApiResponse';
import {EApiEntry} from 'types/EApiEntry';
import IGenericOrderDownloadBlankApiParams from 'server/api/GenericOrderApi/types/downloadBlank/IGenericOrderDownloadBlankApiParams';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from 'server/getContainerConfig';

export class GenericOrderApi extends RestApiClient {
    private readonly serviceTicket: string;
    private readonly mockTrips: boolean;

    constructor({
        logger,
        requestId,
        getApiHost,
        getServiceTicket,
        httpClient,
        rootSpan,
        mockTrips,
    }: IDependencies) {
        super({
            baseURL: `${getApiHost(EApiEntry.TRAVEL)}/generic_booking_flow/v1`,
            logger: logger,
            requestId,
            httpClient,
            rootSpan,
        });

        this.serviceTicket = getServiceTicket('travelAPI');
        this.mockTrips = mockTrips;
    }

    createOrder(
        params: IGenericOrderCreateOrderApiParams,
    ): Promise<IGenericOrderCreateOrderApiResponse> {
        return this.post<IGenericOrderCreateOrderApiResponse>(
            '/create_order',
            params,
        );
    }

    startPayment(params: IGenericOrderStartPaymentApiParams): Promise<void> {
        return this.post<void>('/start_payment', params);
    }

    cancelOrder(params: IGenericOrderCancelOrderApiParams): Promise<void> {
        return this.post<void>('/cancel_order', params);
    }

    getOrderState(
        params: IGenericOrderGetOrderStateApiParams,
    ): Promise<IGenericOrderGetOrderStateApiResponse> {
        return this.get<IGenericOrderGetOrderStateApiResponse>(
            '/get_order_state',
            {params},
        );
    }

    async getOrder(
        params: IGenericOrderGetOrderApiParams,
    ): Promise<IGenericOrderGetOrderApiResponse> {
        const response = await this.get<IGenericOrderGetOrderApiResponse>(
            '/get_order',
            {
                params,
            },
        );

        if (this.mockTrips) {
            response.tripId = '1';
        }

        return response;
    }

    calculateRefundAmount(
        params: IGenericOrderCalculateRefundAmountApiParams,
    ): Promise<IGenericOrderCalculateRefundAmountApiResponse> {
        return this.post<IGenericOrderCalculateRefundAmountApiResponse>(
            '/calculate_refund_amount',
            params,
        );
    }

    startRefund(params: IGenericOrderStartRefundApiParams): Promise<void> {
        return this.post<void>('/start_refund', params);
    }

    estimateDiscount(
        params: IGenericOrderEstimateDiscountApiParams,
    ): Promise<IGenericOrderEstimateDiscountApiResponse> {
        return this.post<IGenericOrderEstimateDiscountApiResponse>(
            '/estimate_discount',
            params,
        );
    }

    addService(
        params: IGenericOrderAddServiceApiParams,
    ): Promise<IGenericOrderAddServiceApiResponse> {
        return this.post<IGenericOrderAddServiceApiResponse>(
            '/add_service',
            params,
        );
    }

    downloadBlank({
        token,
    }: IGenericOrderDownloadBlankApiParams): Promise<NodeJS.ReadableStream> {
        return this.get<NodeJS.ReadableStream>('/download_blank', {
            params: {token},
            responseType: 'stream',
            maxRedirects: 0,
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }
}
