import getRawBody from 'raw-body';
import contentType from 'content-type';

import {CommonHeaders} from '../../constants/headers';
import {PHOTO_UPLOAD_LIMIT} from 'server/constants/hotels';

import {Request} from '@yandex-data-ui/core/lib/types';
import {
    ILegacyHotelInfoRequestParams,
    ILegacyHotelInfoResponse,
} from './types/ILegacyHotelInfo';
import {IHotelInfoRequestParams, IHotelInfoResponse} from './types/IHotelInfo';
import {
    IHotelReviewReactionRequestParams,
    IHotelReviewReactionResponse,
} from './types/IHotelReviewReaction';
import {
    IHotelImagesRequestParams,
    IHotelImagesResponse,
} from './types/IHotelImages';
import {
    IHotelReviewsInfoRequestParams,
    IHotelReviewsInfoResponse,
} from './types/IHotelReviews';
import {
    IAddHotelReviewRequest,
    IAddHotelReviewResponse,
} from './types/IAddHotelReview';
import {
    IEditHotelReviewRequest,
    IEditHotelReviewResponse,
} from './types/IEditHotelReview';
import {
    IHotelOffersInfoRequestParams,
    IHotelOffersInfoResponse,
} from './types/IOffersInfo';
import {EApiEntry} from 'types/EApiEntry';
import {
    ISimilarHotelsInfoRequestParams,
    ISimilarHotelsInfoResponse,
} from './types/ISimilarHotels';
import {
    IDeleteHotelReviewRequest,
    IDeleteHotelReviewResponse,
} from './types/IDeleteHotelReview';
import {IUploadHotelReviewImageResponse} from './types/IUploadHotelReviewImage';
import {
    IDeleteHotelReviewImagesResponse,
    IDeleteHotelReviewImagesRequestQuery,
    IDeleteHotelReviewImagesRequestBody,
} from './types/IDeleteHotelReviewImages';
import {
    ICalendarPricesRequest,
    ICalendarPricesResponse,
} from './types/ICalendarPrices';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from '../../getContainerConfig';

export class HotelSearchAPI extends RestApiClient {
    private readonly serviceTicket: string;

    constructor({
        logger,
        requestId,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        httpClient,
        sendClickHouseStats,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL: `${getApiHost(EApiEntry.TRAVEL)}/hotels_portal/v1`,
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
    }

    getLegacyHotelInfo(
        params: ILegacyHotelInfoRequestParams,
    ): Promise<ILegacyHotelInfoResponse> {
        return this.get<ILegacyHotelInfoResponse>('/get_legacy_hotel/', {
            params,
        });
    }

    getHotelInfo(params: IHotelInfoRequestParams): Promise<IHotelInfoResponse> {
        return this.get<IHotelInfoResponse>('/get_hotel_info/', {
            params,
        });
    }

    getCalendarPrices(
        params: ICalendarPricesRequest,
    ): Promise<ICalendarPricesResponse> {
        return this.get<ICalendarPricesResponse>('/get_calendar_prices/', {
            params,
        });
    }

    getHotelImages(
        imagesParams: IHotelImagesRequestParams,
    ): Promise<IHotelImagesResponse> {
        return this.get<IHotelImagesResponse>('/get_hotel_images/', {
            params: imagesParams,
        });
    }

    getHotelReviews(
        reviewsParams: IHotelReviewsInfoRequestParams,
    ): Promise<IHotelReviewsInfoResponse> {
        return this.get<IHotelReviewsInfoResponse>('/get_hotel_reviews/', {
            params: reviewsParams,
        });
    }

    addHotelReview({
        hotelSlug,
        ...rest
    }: IAddHotelReviewRequest): Promise<IAddHotelReviewResponse> {
        return this.post<IAddHotelReviewResponse>(
            '/add_hotel_review/',
            {
                ...rest,
                ...this.srcParams,
            },
            {
                params: {hotelSlug},
            },
        );
    }

    editHotelReview({
        reviewId,
        hotelSlug,
        ...rest
    }: IEditHotelReviewRequest): Promise<IEditHotelReviewResponse> {
        return this.post<IEditHotelReviewResponse>(
            '/edit_hotel_review/',
            {
                ...rest,
                ...this.srcParams,
            },
            {
                params: {reviewId, hotelSlug},
            },
        );
    }

    deleteHotelReview(
        request: IDeleteHotelReviewRequest,
    ): Promise<IDeleteHotelReviewResponse> {
        return this.post<IDeleteHotelReviewResponse>(
            '/delete_hotel_review/',
            {},
            {
                params: {...request, ...this.srcParams},
            },
        );
    }

    async proxyUploadHotelReviewImage(
        req: Request,
    ): Promise<IUploadHotelReviewImageResponse> {
        const rawData = await getRawBody(req, {
            length: req.headers['content-length'],
            limit: PHOTO_UPLOAD_LIMIT,
            encoding: contentType.parse(req).parameters.charset,
        });

        return this.post<IUploadHotelReviewImageResponse>(
            '/upload_hotel_review_image/',
            rawData,
            {
                headers: {
                    'Content-Type': req.get('Content-Type') ?? '',
                },
                params: {...req.query, ...this.srcParams},
            },
        );
    }

    async deleteHotelReviewImages(
        params: IDeleteHotelReviewImagesRequestQuery,
        request: IDeleteHotelReviewImagesRequestBody,
    ): Promise<IDeleteHotelReviewImagesResponse> {
        return this.post<IUploadHotelReviewImageResponse>(
            '/delete_hotel_review_images/',
            request,
            {
                params: {...params, ...this.srcParams},
            },
        );
    }

    getHotelOffers(
        params: IHotelOffersInfoRequestParams,
    ): Promise<IHotelOffersInfoResponse> {
        return this.get<IHotelOffersInfoResponse>('/get_hotel_offers/', {
            params,
        });
    }

    getSimilarHotels(
        similarHotelsParams: ISimilarHotelsInfoRequestParams,
    ): Promise<ISimilarHotelsInfoResponse> {
        return this.get<ISimilarHotelsInfoResponse>('/get_similar_hotels/', {
            params: similarHotelsParams,
        });
    }

    setHotelReviewReaction(
        reaction: IHotelReviewReactionRequestParams,
    ): Promise<IHotelReviewReactionResponse> {
        return this.post('/set_hotel_review_reaction/', this.srcParams, {
            params: reaction,
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.timeout = 25000;
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }
}
