import {Readable} from 'stream';

import {CommonHeaders} from 'server/constants/headers';

import {IFetchBookOfferParams} from './types/IBookOffer';
import {IBookOfferResponse} from './types/IBookOfferResponse';
import {IOrderParams} from './types/IOrder';
import {IOrderResponse, IOrderStatusResponse} from './types/IOrderResponse';
import {
    ICreateOrderRequestParams,
    ICreateOrderResponse,
} from './types/ICreateOrder';
import {
    IStartPaymentParams,
    TStartPaymentResponse,
} from './types/TStartPayment';
import {
    IOrderRefundableInfoParams,
    IOrderRefundableInfoResponse,
} from './types/IOrderRefundableInfo';
import {ICancelOrderParams, ICancelOrderResponse} from './types/ICancelOrder';
import {IOrderPdfLinkResponse} from './types/IOrderPdfLinkResponse';
import {IStartDownloadVoucherParams} from './types/IDownloadVoucher';
import {
    ITestBookOfferTokenRequestParams,
    ITestBookOfferTokenResponse,
} from './types/ITestBookOfferToken';
import {
    IEstimateDiscountRequestParams,
    IEstimateDiscountResponse,
} from './types/IEstimateDiscount';
import {ICustomerCreateOrderRequest} from 'types/hotels/book/ICreateOrder';
import {EApiEntry} from 'types/EApiEntry';
import IGenerateBusinessTripPdfRequest from 'server/api/HotelsBookAPI/types/IGenerateBusinessTripPdfRequest';
import {IOrderPdfLinkRequest} from 'server/api/HotelsBookAPI/types/IOrderPdfLinkRequest';

import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';
import {RestApiClient} from 'server/utilities/RestApiClient';
/* Prepare request params */
import convertParamsForCreateOrder from './utilities/convertParamsForCreateOrder';
import convertParamsForStartPayment from './utilities/convertParamsForStartPayment';
import convertParamsForGetOrder from './utilities/convertParamsForGetOrder';
import convertParamsForOrderRefundableInfo from './utilities/convertParamsForOrderRefundableInfo';
import convertParamsForCancelOrder from './utilities/convertParamsForCancelOrder';
import convertParamsForOrderVoucherLink from './utilities/convertParamsForOrderVoucherLink';

import {IDependencies} from 'server/getContainerConfig';

export class HotelsBookAPI extends RestApiClient {
    private static getCamelCaseResponseHeaders(): {
        [CommonHeaders.X_YA_USE_CAMELCASE]: 'true';
    } {
        return {
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }

    private readonly serviceTicket: string;
    private readonly mockTrips: boolean;

    constructor({
        logger,
        requestId,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        httpClient,
        sendClickHouseStats,
        rootSpan,
        mockTrips,
    }: IDependencies) {
        super({
            baseURL: getApiHost(EApiEntry.TRAVEL),
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
        this.mockTrips = mockTrips;
    }

    /* Fetch offer flow */

    fetchOfferInfo(
        fetchOfferParams: IFetchBookOfferParams,
    ): Promise<IBookOfferResponse> {
        const {token, label, customerIp, customerUserAgent} = fetchOfferParams;

        return this.get('/booking_flow/v1/get_order_info_by_token', {
            params: {
                token,
                label,
                customerIp,
                customerUserAgent,
            },
            headers: {
                [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
            },
        });
    }

    /* Create order flow */

    createOrder(
        createOrderParams: ICustomerCreateOrderRequest,
    ): Promise<ICreateOrderResponse> {
        const preparedOrderParams: ICreateOrderRequestParams =
            convertParamsForCreateOrder(createOrderParams);

        return this.post('/booking_flow/v1/create_order', preparedOrderParams);
    }

    startPayment(
        startPaymentParams: IStartPaymentParams,
    ): Promise<TStartPaymentResponse> {
        const preparedStartPaymentParams =
            convertParamsForStartPayment(startPaymentParams);

        return this.post(
            '/booking_flow/v1/start_payment',
            preparedStartPaymentParams,
        );
    }

    async getOrder(orderParams: IOrderParams): Promise<IOrderResponse> {
        const preparedOrderRequestParams =
            convertParamsForGetOrder(orderParams);

        const response = await this.get<IOrderResponse>(
            '/booking_flow/v1/get_order',
            {
                params: preparedOrderRequestParams,
                headers: {
                    [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
                },
            },
        );

        if (this.mockTrips) {
            response.tripId = '1';
        }

        return response;
    }

    getOrderStatus(orderParams: IOrderParams): Promise<IOrderStatusResponse> {
        const preparedOrderRequestParams =
            convertParamsForGetOrder(orderParams);

        return this.get('/booking_flow/v1/get_order_status', {
            params: preparedOrderRequestParams,
            headers: {
                [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
            },
        });
    }

    estimateDiscount(
        promoCodesInfo: IEstimateDiscountRequestParams,
    ): Promise<IEstimateDiscountResponse> {
        return this.post('/booking_flow/v1/estimate_discount', promoCodesInfo, {
            headers: HotelsBookAPI.getCamelCaseResponseHeaders(),
        });
    }

    /* Cancel order flow */

    getOrderRefundableInfo(
        orderParams: IOrderRefundableInfoParams,
    ): Promise<IOrderRefundableInfoResponse> {
        const preparedOrderRefundableRequestParams =
            convertParamsForOrderRefundableInfo(orderParams);

        return this.get('/booking_flow/v1/calculate_refund', {
            params: preparedOrderRefundableRequestParams,
        });
    }

    cancelOrder(
        cancelOrderParams: ICancelOrderParams,
    ): Promise<ICancelOrderResponse> {
        const {data, params} = convertParamsForCancelOrder(cancelOrderParams);

        return this.post('/booking_flow/v1/start_refund', data, {params});
    }

    removeOrder(id: string): Promise<void> {
        return this.post('/booking_flow/v1/cancel_order', {id});
    }

    /* Download business trip doc */

    generateBusinessTripPdf(
        params: IGenerateBusinessTripPdfRequest,
    ): Promise<void> {
        return this.get('/booking_flow/v1/generate_business_trip_pdf', {
            params,
        });
    }

    /* Download voucher flow */

    getOrderPdfLink(
        params: IOrderPdfLinkRequest,
    ): Promise<IOrderPdfLinkResponse> {
        const preparedVoucherParams = convertParamsForOrderVoucherLink(params);

        return this.get('/booking_flow/v1/get_order_pdf_link', {
            params: preparedVoucherParams,
            headers: HotelsBookAPI.getCamelCaseResponseHeaders(),
        });
    }

    async downloadPdf(params: IStartDownloadVoucherParams): Promise<Readable> {
        const {url} = params;

        const {data} = await this.httpClient.request({
            url,
            method: 'GET',
            responseType: 'stream',
            maxRedirects: 0,
        });

        return data;
    }

    getTestBookOfferToken(
        offerParams: ITestBookOfferTokenRequestParams,
    ): Promise<ITestBookOfferTokenResponse> {
        return this.get('/test_context/v1/tokens', {
            params: offerParams,
            headers: HotelsBookAPI.getCamelCaseResponseHeaders(),
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.timeout = 40000;
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
        };
    }
}
