import _keys from 'lodash/keys';
import _get from 'lodash/get';
import mapKeys from 'lodash/mapKeys';
import camelcaseKeys from 'camelcase-keys';
import {AxiosError} from 'axios';

import {StatusCodes} from '../constants/statusCodes';
import {EOfferStatus} from 'projects/hotels/constants/EOfferStatus';

import {
    TBookLegalInfoResponse,
    IBasicHotelInfoResponse,
    IBookOfferBadRequestErrorResponse,
    IBookOfferCancellationInfoResponse,
    IBookOfferErrorResponse,
    IBookOfferHotelChargesResponse,
    IBookOfferNotFoundErrorResponse,
    IBookOfferRateResponse,
    IBookOfferResponse,
    IBookRequestInfoResponse,
    IPartnerHotelInfoResponse,
    IBookDeferredPaymentScheduleResponse,
} from '../types/IBookOfferResponse';
import {
    TBookLegalInfo,
    IBookHotelInfo,
    IBookOffer,
    IBookOfferCancellationInfo,
    IBookOfferHotelCharges,
    IBookOfferTotalsCharges,
    IBookOfferMeal,
    IBookOfferPriceInfo,
    IBookOfferRoom,
    IBookOrderOfferInfo,
    IBookPartnerHotelInfo,
    IBookSearchParams,
    IBookOfferInfoError,
    IParsedTokenInfo,
    IBookDeferredPaymentSchedule,
} from '../types/IBookOffer';
import {IBreadcrumbs} from 'reducers/hotels/hotelPage/hotelInfo/types';
import {IHotelOfferMeal} from 'types/hotels/offer/IHotelOfferMeal';
import ICancellationPenalty from 'server/api/GenericOrderApi/types/common/service/IHotelServiceInfo/IHotelOfferInfo/ICancellationInfo/ICancellationPenalty';
import IFees from 'server/api/GenericOrderApi/types/common/service/IHotelServiceInfo/IHotelOfferInfo/IPartnerHotelInfo/IFees';
import IPolicies from 'server/api/GenericOrderApi/types/common/service/IHotelServiceInfo/IHotelOfferInfo/IPartnerHotelInfo/IPolicies';
import IHotelAddress from 'server/api/GenericOrderApi/types/common/service/IHotelServiceInfo/IHotelOfferInfo/IPartnerHotelInfo/IHotelAddress';
import {ICoordinates} from 'server/api/GenericOrderApi/types/common/service/IHotelServiceInfo/IHotelOfferInfo/IPartnerHotelInfo/ICoordinates';

/* Helpers */
const prepareRoomInfo = (roomInfo: IBookOfferRoom): IBookOfferRoom => {
    const {name, images, amenities, descriptions}: IBookOfferRoom =
        roomInfo || {};

    return {
        name,
        images: (images || []).map(image => ({
            ...image,
            links: mapKeys(image.links, (_, key) => key.toLowerCase()),
        })),
        amenities: amenities || {},
        descriptions: descriptions || {},
    };
};

const prepareMealInfo = (mealInfo: IHotelOfferMeal): IBookOfferMeal => {
    const {id, name}: IHotelOfferMeal = mealInfo || {};

    return {
        id,
        name,
    };
};

const preparePriceTotals = (
    totals: IBookOfferHotelChargesResponse['totals'],
): IBookOfferTotalsCharges => {
    const {
        base: hotelBase,
        grand: totalPrice,
        taxesAndFees,
        taxesAndFeesSum,
        strikeThrough,
        discount,
        priceAfterPlusWithdraw,
    }: IBookOfferHotelChargesResponse['totals'] = totals || {};

    return {
        hotelBase,
        totalPrice,
        taxesAndFees: taxesAndFees || [],
        taxesAndFeesSum: taxesAndFeesSum || {},
        strikeThrough,
        discount,
        priceAfterPlusWithdraw,
    };
};

const preparePriceNightly = (
    nightly: IBookOfferHotelChargesResponse['nightly'],
): IBookOfferHotelCharges['nightly'] =>
    nightly.map(({base: hotelBase, taxesAndFees}) => ({
        hotelBase: hotelBase || {},
        taxesAndFees: taxesAndFees || [],
    }));

const preparePriceInfo = (
    rateInfo: IBookOfferRateResponse,
): IBookOfferPriceInfo => {
    const {hotelCharges, extraCharges}: IBookOfferRateResponse = rateInfo || {};
    const {nightly, totals}: IBookOfferHotelChargesResponse =
        hotelCharges || {};

    if (nightly && totals) {
        return {
            hotelCharges: {
                totals: preparePriceTotals(totals || {}),
                nightly: preparePriceNightly(nightly || []),
            },
            extraCharges: extraCharges || [],
        };
    }

    return {
        hotelCharges: {},
        extraCharges: [],
    };
};

const preparePartnerHotelInfo = (
    partnerHotelInfo: IPartnerHotelInfoResponse,
): IBookPartnerHotelInfo => {
    const {
        name: hotelName,
        category,
        ratings,
        descriptions,
        address,
        amenities: amenitiesById,
        location: coordinatesLocation,
        images,
    }: IPartnerHotelInfoResponse = partnerHotelInfo || {};

    const {longitude: lng, latitude: lat} = _get(
        coordinatesLocation,
        'coordinates',
        {},
    );
    const {name: hotelType}: IPartnerHotelInfoResponse['category'] =
        category || {};
    const {property: propertyRating}: IPartnerHotelInfoResponse['ratings'] =
        ratings || {};
    const {
        location,
        rooms,
        dining,
        amenities,
        businessAmenities,
    }: IPartnerHotelInfoResponse['descriptions'] = descriptions || {};

    const {
        city,
        countryCode,
        line1: firstLine,
        line2: secondLine,
    }: IHotelAddress = address || {};

    return {
        images: images || [],
        hotelName,
        hotelType,
        amenities: amenitiesById || {},
        propertyRating: propertyRating || {},
        descriptions: {
            location,
            rooms,
            dining,
            amenities,
            businessAmenities,
        },
        coordinates: {lat, lng},
        address: {
            city,
            countryCode,
            firstLine,
            secondLine,
        },
    };
};

const prepareHotelInfo = (
    basicHotelInfo: IBasicHotelInfoResponse,
    partnerHotelInfo: IPartnerHotelInfoResponse,
): IBookHotelInfo => {
    const {
        address,
        coordinates,
        imageUrlTemplate,
        name,
        rating,
        stars,
        permalink,
        hotelSlug,
        breadcrumbs,
    }: IBasicHotelInfoResponse = basicHotelInfo || {};

    const {longitude: lng, latitude: lat}: ICoordinates = coordinates || {};

    const {
        checkin,
        checkout,
        fees,
        policies,
        name: partnerHotelName,
        phone,
        images,
    }: IPartnerHotelInfoResponse = partnerHotelInfo || {};

    const {
        instructions,
        beginTime: checkinBeginTime,
        endTime: checkinEndTime,
        specialInstructions: specialInstructions,
    }: IPartnerHotelInfoResponse['checkin'] = checkin || {};
    const {time: checkoutEndTime}: IPartnerHotelInfoResponse['checkout'] =
        checkout || {};
    const {mandatory: mandatoryCharge, optional: optionalCharge}: IFees =
        fees || {};
    const {knowBeforeYouGo: policyCheckinInfo}: IPolicies = policies || {};

    return {
        checkin: {
            beginTime: checkinBeginTime,
            endTime: checkinEndTime,
        },
        checkout: {
            endTime: checkoutEndTime,
        },
        settlementInfo: {
            instructions,
            optionalCharge,
            mandatoryCharge,
            policyCheckinInfo,
            specialInstructions,
        },
        name,
        phone,
        images: images || [],
        rating,
        stars,
        address,
        hotelSlug,
        permalink,
        partnerHotelName,
        breadcrumbs: breadcrumbs
            ? (camelcaseKeys(breadcrumbs, {
                  deep: true,
              }) as unknown as IBreadcrumbs)
            : undefined,
        imageUrlTemplate: imageUrlTemplate || '',
        coordinates: {lat, lng},
    };
};

const prepareLegalInfo = (legalInfo: TBookLegalInfoResponse): TBookLegalInfo =>
    _keys(legalInfo || {}).map(source => {
        const {
            name,
            ogrn,
            registryNumber,
            legalAddress,
            actualAddress,
            workingHours,
        } = legalInfo[source];

        return {
            source,
            name,
            ogrn,
            registryNumber,
            workingHours,
            legalAddress,
            actualAddress,
        };
    });

const prepareSearchParams = (
    requestInfo: IBookRequestInfoResponse,
): IBookSearchParams => {
    const {
        checkinDate: startDate,
        checkoutDate: endDate,
        numAdults: adults,
        childAges: childrenAges,
    }: IBookRequestInfoResponse = requestInfo || {};

    return {
        endDate,
        startDate,
        adults,
        childrenAges: childrenAges || [],
    };
};

const prepareCancellationInfo = (
    cancellationInfo?: IBookOfferCancellationInfoResponse,
): IBookOfferCancellationInfo => {
    if (cancellationInfo) {
        const {refundable, penalties, highlighted} = cancellationInfo;

        const refundableRules = penalties.map(penalty => {
            const {
                type,
                amount,
                currency,
                endsAt: endDate,
                startsAt: startDate,
            }: ICancellationPenalty = penalty || {};

            return {
                price: {
                    amount,
                    currency,
                },
                type,
                endDate,
                startDate,
            };
        });

        return {
            refundable,
            refundableRules,
            highlighted,
        };
    }

    return {
        refundable: false,
        highlighted: false,
        refundableRules: [],
    };
};

const prepareOfferOrderInfo = (
    offerData: IBookOfferResponse,
): IBookOrderOfferInfo => {
    const {
        label,
        checksum,
        sessionKey,
        travelToken: token,
    }: IBookOfferResponse = offerData || {};

    return {
        label,
        checksum,
        sessionKey,
        token,
    };
};

export const prepareDeferredPaymentSchedule = (
    deferredPaymentScheduleResponse?: IBookDeferredPaymentScheduleResponse | null,
): IBookDeferredPaymentSchedule | undefined => {
    if (!deferredPaymentScheduleResponse) {
        return undefined;
    }

    return deferredPaymentScheduleResponse;
};

export const prepareOfferInfo = (
    bookOfferInfoResponse: IBookOfferResponse,
): IBookOffer => {
    const {
        partnerId,
        partnerHotelInfo,
        partnerRoomInfo: roomInfo,
        basicHotelInfo,
        bedGroups: bedsGroups,
        requestInfo,
        cancellationInfo,
        cancellationInfoUnfiltered,
        legalInfo,
        pansionInfo: mealInfo,
        rateInfo,
        refundInfo,
        directPartner,
        promoCampaigns,
        deferredPaymentSchedule,
        extraVisitAndUserParams,
        allGuestsRequired,
    } = bookOfferInfoResponse;

    return {
        partnerId,
        bedsGroups: bedsGroups || [],
        refundableInfo: refundInfo,
        mealInfo: prepareMealInfo(mealInfo),
        offerOrderInfo: prepareOfferOrderInfo(bookOfferInfoResponse),
        priceInfo: preparePriceInfo(rateInfo),
        cancellationInfo: prepareCancellationInfo(cancellationInfo),
        cancellationInfoAll: prepareCancellationInfo(
            cancellationInfoUnfiltered,
        ),
        partnerHotelInfo: preparePartnerHotelInfo(partnerHotelInfo),
        roomInfo: prepareRoomInfo(roomInfo),
        hotelInfo: prepareHotelInfo(basicHotelInfo, partnerHotelInfo),
        legalInfo: prepareLegalInfo(legalInfo),
        searchParams: prepareSearchParams(requestInfo),
        directPartner,
        promoCampaigns,
        deferredPaymentSchedule: prepareDeferredPaymentSchedule(
            deferredPaymentSchedule,
        ),
        extraVisitAndUserParams,
        allGuestsRequired,
    };
};

export const prepareTokenParams = (
    errorResponse?: IBookOfferNotFoundErrorResponse,
): IParsedTokenInfo | undefined => {
    const {token} = errorResponse || {};

    if (token) {
        const {
            originalId,
            checkInDate: checkinDate,
            checkOutDate: checkoutDate,
            permalink,
            partnerId,
            generatedAt,
            offerId,
            tokenId,
            occupancy,
        } = token;

        return {
            originalId,
            checkinDate,
            checkoutDate,
            permalink,
            partnerId,
            generatedAt,
            offerId,
            tokenId,
            occupancy,
        };
    }
};

export const getOfferInfoError = ({
    status,
    data,
}: {
    status: StatusCodes | undefined;
    data?:
        | IBookOfferResponse
        | IBookOfferErrorResponse
        | IBookOfferNotFoundErrorResponse
        | IBookOfferBadRequestErrorResponse;
}): IBookOfferInfoError => {
    switch (status) {
        case StatusCodes.OFFER_EXPIRED: {
            return {
                error: {
                    code: EOfferStatus.OFFER_IS_NOT_AVAILABLE,
                    offerInfo: prepareOfferInfo(data as IBookOfferResponse),
                },
                statusCode: StatusCodes.OFFER_EXPIRED,
            };
        }

        case StatusCodes.OFFER_BAD_REQUEST: {
            return {
                error: {
                    code: EOfferStatus.WRONG_OFFER_TOKEN,
                },
                statusCode: StatusCodes.OFFER_BAD_REQUEST,
            };
        }

        case StatusCodes.OFFER_NOT_FOUND: {
            return {
                error: {
                    code: EOfferStatus.OFFER_NOT_FOUND_ERROR,
                    tokenInfo: prepareTokenParams(
                        data as IBookOfferNotFoundErrorResponse,
                    ),
                },
                statusCode: StatusCodes.OFFER_NOT_FOUND,
            };
        }

        default: {
            return {
                error: {
                    code: EOfferStatus.UNKNOWN_OFFER_ERROR,
                },
                statusCode: StatusCodes.OFFER_UNKNOWN_ERROR,
            };
        }
    }
};

export const prepareOfferInfoError = (
    apiError: AxiosError,
): IBookOfferInfoError => {
    const {response} = apiError;

    const status = response?.status as StatusCodes | undefined;
    const data = response?.data as
        | IBookOfferResponse
        | IBookOfferErrorResponse
        | IBookOfferNotFoundErrorResponse
        | IBookOfferBadRequestErrorResponse
        | undefined;

    return getOfferInfoError({status, data});
};
