/* External utilities */
import _get from 'lodash/get';
import _isObject from 'lodash/isObject';
/* Constants */

import {StatusCodes} from '../constants/statusCodes';

/* Types */
import {
    IOrder,
    IOrderCancellationDetails,
    IOrderConfirmationInfo,
    IOrderError,
    IOrderGuestsInfo,
    IOrderPayment,
    IOrderRefundableInfo,
    IOrderPriceInfo,
} from '../types/IOrder';
import {
    IOrderCancellationDetailsResponse,
    IOrderConfirmationInfoResponse,
    IOrderGuestsInfoResponse,
    IOrderPaymentResponse,
    IOrderRefundInfoResponse,
    IOrderResponse,
    IOrderPriceInfoResponse,
} from '../types/IOrderResponse';
import {
    IBookOfferResponse,
    IBookRequestInfoResponse,
} from '../types/IBookOfferResponse';
import {IBookOffer} from '../types/IBookOffer';

/* Utilities */
import {prepareOfferInfo} from './prepareOfferInfo';

const preparePaymentInfo = (payment?: IOrderPaymentResponse): IOrderPayment => {
    if (payment) {
        const {
            errorInfo: errorCode,
            paymentUrl: paymentUrl,
            next,
            error,
            current,
            receipts,
            amountPaid,
            mayBeStarted,
            mayBeCancelled,
            usesDeferredPayments,
            usesZeroFirstPayment,
        } = payment;

        return {
            next,
            error,
            current,
            receipts,
            errorCode,
            paymentUrl,
            amountPaid,
            mayBeStarted,
            mayBeCancelled,
            usesDeferredPayments,
            usesZeroFirstPayment,
        };
    }

    return {};
};

const prepareRefundableInfo = (
    refundableInfo?: IOrderRefundInfoResponse,
): IOrderRefundableInfo => {
    if (refundableInfo) {
        const {penalty, refund, penaltyIntervalIndex, refundDateTime} =
            refundableInfo;

        const {
            amount: penaltyAmount,
            currency: penaltyCurrency,
        }: IOrderRefundInfoResponse['penalty'] = penalty || {};
        const {
            amount: refundAmount,
            currency: refundCurrency,
        }: IOrderRefundInfoResponse['refund'] = refund || {};

        return {
            penalty: {
                amount: penaltyAmount,
                currency: penaltyCurrency,
            },
            refund: {
                amount: refundAmount,
                currency: refundCurrency,
            },
            penaltyIntervalIndex,
            refundDateTime,
        };
    }

    return {};
};

export const prepareGuestsInfo = ({
    guestsInfo,
    requestInfo,
}: {
    guestsInfo: IOrderGuestsInfoResponse;
    requestInfo: IBookRequestInfoResponse;
}): IOrderGuestsInfo => {
    const {customerEmail, customerPhone, guests} = guestsInfo;

    const {selectedBedGroupIndex} = requestInfo;

    return {
        customerEmail,
        customerPhone,
        selectedBedGroupIndex,
        guests: guests.map(({firstName, lastName}) => ({
            firstName,
            lastName,
        })),
    };
};

const prepareConfirmationInfo = ({
    yandexOrderId,
    confirmationInfo,
}: {
    yandexOrderId: string;
    confirmationInfo: IOrderConfirmationInfoResponse;
}): IOrderConfirmationInfo => {
    const {confirmationId}: IOrderConfirmationInfoResponse =
        confirmationInfo || {};

    return {
        yandexOrderId,
        confirmationId,
    };
};

const prepareOrderOfferInfo = (offerInfo: IBookOfferResponse): IBookOffer => {
    return prepareOfferInfo(offerInfo);
};

const prepareOrderPriceInfo = (
    info?: IOrderPriceInfoResponse,
): IOrderPriceInfo | null => {
    if (!info) {
        return null;
    }

    const {
        price,
        originalPrice,
        discountAmount,
        promoCampaigns,
        promoCodeApplicationResults: codeApplicationResults,
    } = info;

    return {
        price,
        originalPrice,
        discountAmount,
        promoCampaigns,
        codeApplicationResults: codeApplicationResults || [],
    };
};

function prepareCancellationDetails(
    orderCancellationDetails?: IOrderCancellationDetailsResponse | null,
): IOrderCancellationDetails | {} {
    if (!_isObject(orderCancellationDetails)) {
        return {};
    }

    return {
        reason: orderCancellationDetails.reason,
        invalidInputDetails: (
            orderCancellationDetails?.invalidInputDetails || []
        ).map(item => {
            return {
                fieldType: item.fieldType,
            };
        }),
    };
}

export const prepareOrderResponse = (orderResponse: IOrderResponse): IOrder => {
    const {
        id: orderId,
        status,
        displayState,
        payment,
        yandexOrderId,
        confirmationInfo,
        guestInfo: guestsInfo,
        orderCancellationDetails,
        orderInfo,
        refundInfo: refundableInfo,
        orderPriceInfo,
        appliedPromoCampaigns,
        tripId,
        canGenerateBusinessTripDoc,
    } = orderResponse;
    const {requestInfo} = orderInfo;

    return {
        orderId,
        status,
        displayState,
        payment: preparePaymentInfo(payment),
        guestsInfo: prepareGuestsInfo({guestsInfo, requestInfo}),
        refundableInfo: prepareRefundableInfo(refundableInfo),
        confirmationInfo: prepareConfirmationInfo({
            yandexOrderId,
            confirmationInfo,
        }),
        orderCancellationDetails: prepareCancellationDetails(
            orderCancellationDetails,
        ),
        offerInfo: prepareOrderOfferInfo(orderInfo),
        orderPriceInfo: prepareOrderPriceInfo(orderPriceInfo),
        appliedPromoCampaigns,
        tripId,
        canGenerateBusinessTripDoc,
    };
};

export const prepareOrderError = (apiError: unknown): IOrderError => {
    const responseStatusCode = _get(apiError, 'response.status');

    return {
        statusCode:
            responseStatusCode || StatusCodes.UNKNOWN_ORDER_ERROR_STATUS_CODE,
    };
};
