import {CommonHeaders} from 'server/constants/headers';

import {EApiEntry} from 'types/EApiEntry';
import {IHotelsExtranetOrdersResponse} from './types/IHotelsExtranetOrdersResponse';
import {IHotelsExtranetOrdersRequest} from './types/IHotelsExtranetOrdersRequest';
import {IHotelsExtranetPaymentsRequest} from './types/IHotelsExtranetPaymentsRequest';
import {IHotelsExtranetPaymentsResponse} from './types/IHotelsExtranetPaymentsResponse';
import {IHotelsExtranetAddHotelRequest} from './types/IHotelsExtranetAddHotelRequest';
import {IHotelsExtranetAddHotelResponse} from './types/IHotelsExtranetAddHotelResponse';
import {IHotelsExtranetApproveHotelRequest} from './types/IHotelsExtranetApproveHotelRequest';
import {IHotelsExtranetApproveHotelResponse} from './types/IHotelsExtranetApproveHotelResponse';
import {IHotelsListResponse} from './types/IHotelsListResponse';
import {IHotelsExtranetPaymentDetailsRequest} from './types/IHotelsExtranetPaymentDetailsRequest';
import {IHotelsExtranetPaymentDetailsResponse} from './types/IHotelsExtranetPaymentDetailsResponse';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from 'server/getContainerConfig';

export class HotelsExtranetApi extends RestApiClient {
    private serviceTicket: string;

    constructor({
        logger,
        requestId,
        getApiHost,
        httpClient,
        sendClickHouseStats,
        getServiceTicket,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL: getApiHost(EApiEntry.HOTELS_EXTRANET),
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.serviceTicket = getServiceTicket('travelAPI');
    }

    addHotel(
        request: IHotelsExtranetAddHotelRequest,
    ): Promise<IHotelsExtranetAddHotelResponse> {
        return this.post<IHotelsExtranetAddHotelResponse>(
            '/auth/v1/request/',
            request,
        );
    }

    approveHotel(
        request: IHotelsExtranetApproveHotelRequest,
    ): Promise<IHotelsExtranetApproveHotelResponse> {
        return this.post<IHotelsExtranetApproveHotelResponse>(
            '/auth/v1/approve/',
            request,
        );
    }

    getHotels(): Promise<IHotelsListResponse> {
        return this.get<IHotelsListResponse>('/auth/v1/list_approved/');
    }

    getOrders(
        request: IHotelsExtranetOrdersRequest,
    ): Promise<IHotelsExtranetOrdersResponse> {
        return this.post<IHotelsExtranetOrdersResponse>(
            '/v1/get_orders/',
            request,
        );
    }

    getPayments(
        request: IHotelsExtranetPaymentsRequest,
    ): Promise<IHotelsExtranetPaymentsResponse> {
        return this.post<IHotelsExtranetPaymentsResponse>(
            '/v1/get_bank_orders/',
            request,
        );
    }

    getPaymentDetails(
        params: IHotelsExtranetPaymentDetailsRequest,
    ): Promise<IHotelsExtranetPaymentDetailsResponse> {
        return this.get<IHotelsExtranetPaymentDetailsResponse>(
            `/v1/bank_orders/${params.paymentBatchId}`,
        );
    }

    protected interceptRequest(request: IRequestConfig): void {
        request.timeout = 25000;
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }
}
