import {CommonHeaders} from 'server/constants/headers';

import {EApiEntry} from 'types/EApiEntry';
import IAddFavoriteHotelRequestParams from './types/IAddFavoriteHotelRequestParams';
import {TRemoveFavoriteHotelsRequestParams} from './types/IRemoveFavoriteHotelsRequestParams';
import {TAddFavoriteHotelResponse} from './types/TAddFavoriteHotelResponse';
import IRemoveFavoriteHotelsResponse from './types/IRemoveFavoriteHotelsResponse';
import IGetFavoriteHotelsRequestParams from './types/IGetFavoriteHotelsRequestParams';
import IGetFavoriteHotelsResponse from './types/IGetFavoriteHotelsResponse';
import IGetFavoriteHotelsOffersRequestParams from './types/IGetFavoriteHotelsOffersRequestParams';
import IGetFavoriteHotelsOffersResponse from './types/IGetFavoriteHotelsOffersResponse';
import IShareFavoriteHotelsResponse from './types/IShareFavoriteHotelsResponse';
import IShareFavoriteHotelsRequestParams from './types/IShareFavoriteHotelsRequestParams';
import IGetSharedFavoriteHotelsRequestParams from './types/IGetSharedFavoriteHotelsRequestParams';
import IGetSharedFavoriteHotelsResponse from './types/IGetSharedFavoriteHotelsResponse';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from '../../getContainerConfig';

export class HotelsFavoritesAPI extends RestApiClient {
    private readonly serviceTicket: string;

    constructor({
        logger,
        requestId,
        getApiHost,
        httpClient,
        sendClickHouseStats,
        getServiceTicket,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL: `${getApiHost(EApiEntry.TRAVEL)}/hotels_portal/v1`,
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.serviceTicket = getServiceTicket('travelAPI');
    }

    addFavoriteHotel(
        params: IAddFavoriteHotelRequestParams,
    ): Promise<TAddFavoriteHotelResponse> {
        return this.post<TAddFavoriteHotelResponse>(
            `/add_favorite_hotel`,
            undefined,
            {params},
        );
    }

    removeFavoriteHotels(
        params: TRemoveFavoriteHotelsRequestParams,
    ): Promise<IRemoveFavoriteHotelsResponse> {
        return this.post<IRemoveFavoriteHotelsResponse>(
            '/remove_favorite_hotels',
            undefined,
            {params},
        );
    }

    getFavoriteHotels(
        params: IGetFavoriteHotelsRequestParams,
    ): Promise<IGetFavoriteHotelsResponse> {
        return this.get<IGetFavoriteHotelsResponse>('/get_favorite_hotels', {
            params,
        });
    }

    getFavoriteHotelsOffers(
        params: IGetFavoriteHotelsOffersRequestParams,
    ): Promise<IGetFavoriteHotelsOffersResponse> {
        return this.get<IGetFavoriteHotelsOffersResponse>(
            '/get_favorite_hotels_offers',
            {params},
        );
    }

    shareFavoriteHotels(
        params: IShareFavoriteHotelsRequestParams,
    ): Promise<IShareFavoriteHotelsResponse> {
        return this.post<IShareFavoriteHotelsResponse>(
            '/share_favorite_hotels',
            undefined,
            {params},
        );
    }

    getSharedFavoriteHotels(
        params: IGetSharedFavoriteHotelsRequestParams,
    ): Promise<IGetSharedFavoriteHotelsResponse> {
        return this.get<IGetSharedFavoriteHotelsResponse>(
            '/get_shared_favorite_hotels',
            {params},
        );
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.timeout = 25000;
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }
}
