import {CommonHeaders} from 'server/constants/headers';

/* SearchHotels */
import {
    ISearchHotelsInfoRequestParams,
    ISearchHotelsInfoResponse,
} from './types/ISearchHotels';
/* Suggest */
import {
    ISearchSuggestGroupsResponse,
    ISearchSuggestRequestParams,
} from './types/ISearchSuggest';
/* LogSuggest */
import {
    ILogSearchSuggestRequestParams,
    ILogSearchSuggestResponse,
} from './types/ILogSearchSuggest';
/* CountHotels */
import {
    ICountHotelsRequestParams,
    ICountHotelsResponse,
} from './types/ICountHotels';
/* GeoRegion */
import {
    IGeoRegionInfoRequestParams,
    IGeoRegionInfoResponse,
} from 'server/api/HotelsSearchAPI/types/IGeoRegionInfo';
import {EApiEntry} from 'types/EApiEntry';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from 'server/getContainerConfig';

export class HotelsSearchAPI extends RestApiClient {
    private readonly serviceTicket: string;

    constructor({
        logger,
        requestId,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        httpClient,
        sendClickHouseStats,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL: `${getApiHost(EApiEntry.TRAVEL)}/hotels_portal/v1`,
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
    }

    searchSuggest(
        suggestParams: ISearchSuggestRequestParams,
    ): Promise<ISearchSuggestGroupsResponse> {
        return this.get<ISearchSuggestGroupsResponse>('/suggest/', {
            headers: this._getUseCamelcaseHeaders(),
            params: suggestParams,
        });
    }

    logSearchSuggest(
        logSearchSuggestParams: ILogSearchSuggestRequestParams,
    ): Promise<ILogSearchSuggestResponse> {
        return this.post<ILogSearchSuggestResponse>(
            '/log_suggest/',
            undefined,
            {
                headers: this._getUseCamelcaseHeaders(),
                params: logSearchSuggestParams,
            },
        );
    }

    searchHotels(
        params: ISearchHotelsInfoRequestParams,
    ): Promise<ISearchHotelsInfoResponse> {
        return this.get<ISearchHotelsInfoResponse>('/search_hotels/', {
            headers: this._getUseCamelcaseHeaders(),
            params,
        });
    }

    countHotels(
        countParams: ICountHotelsRequestParams,
    ): Promise<ICountHotelsResponse> {
        return this.get<ICountHotelsResponse>('/count_hotels/', {
            headers: this._getUseCamelcaseHeaders(),
            params: countParams,
        });
    }

    getGeoRegionInfo(
        regionParams: IGeoRegionInfoRequestParams,
    ): Promise<IGeoRegionInfoResponse> {
        return this.get<IGeoRegionInfoResponse>('/city_static_page/', {
            headers: this._getUseCamelcaseHeaders(),
            params: regionParams,
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.timeout = 25000;
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
        };
    }

    // TODO: 2020-02-13 Убрать после полного перехода на camelCase (HOTELS-4639)
    private _getUseCamelcaseHeaders(): {
        [CommonHeaders.X_YA_USE_CAMELCASE]: 'true';
    } {
        return {
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }
}
