import {CommonHeaders} from 'server/constants/headers';
import {
    LEVEL_TRAVEL_ACCEPT_HEADER,
    LEVEL_TRAVEL_API_KEY,
} from 'server/constants/tours';
import {SECOND} from 'utilities/dateUtils/constants';

import {EApiEntry} from 'types/EApiEntry';
import {
    IAutocompleteParams,
    IGetAvailableCountriesParams,
    IGetCountriesParams,
    IGetCountriesResponse,
    IGetDeparturesParams,
    IGetDeparturesResponse,
    IGetFlightsAndNightsParams,
    IGetFlightsAndNightsResponse,
    IGetRedirectParams,
    IGetRedirectResponse,
    TAutocompleteResponse,
    TGetAvailableCountriesResponse,
} from 'server/api/LevelTravelApi/types';

import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';
import {RestApiClient} from 'server/utilities/RestApiClient';

import {IDependencies} from 'server/getContainerConfig';

export class LevelTravelApi extends RestApiClient {
    private readonly isSSR: boolean;

    constructor({
        logger,
        httpClient,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        requestId,
        rootSpan,
        isSSR,
    }: IDependencies) {
        super({
            baseURL: getApiHost(EApiEntry.LEVEL_TRAVEL),
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.LEVEL_TRAVEL);
        this.isSSR = isSSR;
    }

    async getFlightsAndNights(
        query: IGetFlightsAndNightsParams,
    ): Promise<IGetFlightsAndNightsResponse> {
        return this.get('/references/flights_and_nights', {params: query});
    }

    async getDepartures(
        query: IGetDeparturesParams,
    ): Promise<IGetDeparturesResponse> {
        return this.get('/references/departures', {
            params: query,
            timeout: this.isSSR ? 2 * SECOND : undefined,
        });
    }

    async getCountries(
        query: IGetCountriesParams,
    ): Promise<IGetCountriesResponse> {
        return this.get('/references/countries', {params: query});
    }

    async getAvailableCountries(
        query: IGetAvailableCountriesParams,
    ): Promise<TGetAvailableCountriesResponse> {
        return this.get('/references/available_countries', {params: query});
    }

    async autocomplete(
        query: IAutocompleteParams,
    ): Promise<TAutocompleteResponse> {
        return this.get('/references/autocomplete', {params: query});
    }

    async getRedirect(
        query: IGetRedirectParams,
    ): Promise<IGetRedirectResponse> {
        return this.get('/search/land', {params: query});
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.headers = {
            ...request.headers,
            [CommonHeaders.AUTHORIZATION]: `Token token="${LEVEL_TRAVEL_API_KEY}"`,
            [CommonHeaders.ACCEPT]: LEVEL_TRAVEL_ACCEPT_HEADER,
        };
    }
}
