import snakeCaseKeys from 'snakecase-keys';

import {CommonHeaders} from 'server/constants/headers';

import {EApiEntry} from 'types/EApiEntry';
import {
    INotebookGetTravelerApiParams,
    INotebookGetTravelerApiResponse,
} from 'server/api/NotebookApi/types/getTraveler';
import {
    INotebookGetPassengersApiParams,
    TNotebookGetPassengersApiResponse,
} from 'server/api/NotebookApi/types/getPassengers';
import {
    INotebookGetPassengerApiParams,
    TNotebookGetPassengerApiResponse,
} from 'server/api/NotebookApi/types/getPassenger';
import {
    INotebookGetPassengerDocumentsApiParams,
    TNotebookGetPassengerDocumentsApiResponse,
} from 'server/api/NotebookApi/types/getPassengerDocuments';
import {
    INotebookGetPassengerDocumentApiParams,
    TNotebookGetPassengerDocumentApiResponse,
} from 'server/api/NotebookApi/types/getPassengerDocument';
import {
    INotebookGetPassengerBonusCardsApiParams,
    TNotebookGetPassengerBonusCardsApiResponse,
} from 'server/api/NotebookApi/types/getPassengerBonusCards';
import {
    INotebookGetPassengerBonusCardApiParams,
    TNotebookGetPassengerBonusCardApiResponse,
} from 'server/api/NotebookApi/types/getPassengerBonusCard';
import {TNotebookGetDocumentTypesApiResponse} from 'server/api/NotebookApi/types/getDocumentTypes';
import {
    INotebookCreateOrUpdateTravelerApiParams,
    INotebookCreateOrUpdateTravelerApiResponse,
} from 'server/api/NotebookApi/types/createOrUpdateTraveler';
import {
    INotebookCreatePassengerApiParams,
    TNotebookCreatePassengerApiResponse,
} from 'server/api/NotebookApi/types/createPassenger';
import {
    INotebookEditPassengerApiParams,
    TNotebookEditPassengerApiResponse,
} from 'server/api/NotebookApi/types/editPassenger';
import {
    INotebookDeletePassengerApiParams,
    TNotebookDeletePassengerApiResponse,
} from 'server/api/NotebookApi/types/deletePassenger';
import {
    INotebookEditPassengerDocumentApiParams,
    TNotebookEditPassengerDocumentApiResponse,
} from 'server/api/NotebookApi/types/editPassengerDocument';
import {
    INotebookCreatePassengerDocumentApiParams,
    TNotebookCreatePassengerDocumentApiResponse,
} from 'server/api/NotebookApi/types/createPassengerDocument';
import {
    INotebookDeletePassengerDocumentApiParams,
    TNotebookDeletePassengerDocumentApiResponse,
} from 'server/api/NotebookApi/types/deletePassengerDocument';
import {
    INotebookCreatePassengerBonusCardApiParams,
    TNotebookCreatePassengerBonusCardApiResponse,
} from 'server/api/NotebookApi/types/createPassengerBonusCard';
import {
    INotebookEditPassengerBonusCardApiParams,
    TNotebookEditPassengerBonusCardApiResponse,
} from 'server/api/NotebookApi/types/editPassengerBonusCard';
import {
    INotebookDeletePassengerBonusCardApiParams,
    TNotebookDeletePassengerBonusCardApiResponse,
} from 'server/api/NotebookApi/types/deletePassengerBonusCard';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from 'server/getContainerConfig';

export class NotebookApi extends RestApiClient {
    private readonly userTicket: string;
    private readonly serviceTicket: string;

    constructor({
        logger,
        requestId,
        httpClient,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        blackbox,
        rootSpan,
    }: IDependencies) {
        super({
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            baseURL: `${getApiHost(EApiEntry.TRAVELLERS)}/v1`,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVELLERS);
        this.userTicket = blackbox.userTicket;
        this.serviceTicket = getServiceTicket('travellersAPI');
    }

    getTraveler({
        uid,
        ...params
    }: INotebookGetTravelerApiParams): Promise<INotebookGetTravelerApiResponse> {
        return this.get<INotebookGetTravelerApiResponse>(`/travelers/${uid}`, {
            params: snakeCaseKeys(params as any, {deep: true}),
            convertResponseKeysToCamelCase: true,
        });
    }

    createOrUpdateTraveler({
        uid,
        traveler,
    }: INotebookCreateOrUpdateTravelerApiParams): Promise<INotebookCreateOrUpdateTravelerApiResponse> {
        return this.post<INotebookCreateOrUpdateTravelerApiResponse>(
            `/travelers/${uid}`,
            snakeCaseKeys(traveler as any, {deep: true}),
            {
                convertResponseKeysToCamelCase: true,
            },
        );
    }

    getPassengers({
        uid,
        ...params
    }: INotebookGetPassengersApiParams): Promise<TNotebookGetPassengersApiResponse> {
        return this.get<TNotebookGetPassengersApiResponse>(
            `/travelers/${uid}/passengers`,
            {
                params: snakeCaseKeys(params as any, {deep: true}),
                convertResponseKeysToCamelCase: true,
            },
        );
    }

    getPassenger({
        uid,
        passengerId,
        ...params
    }: INotebookGetPassengerApiParams): Promise<TNotebookGetPassengerApiResponse> {
        return this.get<TNotebookGetPassengerApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}`,
            {
                params: snakeCaseKeys(params as any, {deep: true}),
                convertResponseKeysToCamelCase: true,
            },
        );
    }

    createPassenger({
        uid,
        passenger,
    }: INotebookCreatePassengerApiParams): Promise<TNotebookCreatePassengerApiResponse> {
        return this.post<TNotebookCreatePassengerApiResponse>(
            `/travelers/${uid}/passengers`,
            snakeCaseKeys(passenger as any, {deep: true}),
            {convertResponseKeysToCamelCase: true},
        );
    }

    editPassenger({
        uid,
        passengerId,
        passenger,
    }: INotebookEditPassengerApiParams): Promise<TNotebookEditPassengerApiResponse> {
        return this.put<TNotebookEditPassengerApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}`,
            snakeCaseKeys(passenger as any, {deep: true}),
            {convertResponseKeysToCamelCase: true},
        );
    }

    deletePassenger({
        uid,
        passengerId,
    }: INotebookDeletePassengerApiParams): Promise<TNotebookDeletePassengerApiResponse> {
        return this.delete<TNotebookDeletePassengerApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}`,
            {convertResponseKeysToCamelCase: true},
        );
    }

    getPassengerDocuments({
        uid,
        passengerId,
    }: INotebookGetPassengerDocumentsApiParams): Promise<TNotebookGetPassengerDocumentsApiResponse> {
        return this.get<TNotebookGetPassengerDocumentsApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/documents`,
            {convertResponseKeysToCamelCase: true},
        );
    }

    getPassengerDocument({
        uid,
        passengerId,
        documentId,
    }: INotebookGetPassengerDocumentApiParams): Promise<TNotebookGetPassengerDocumentApiResponse> {
        return this.get<TNotebookGetPassengerDocumentApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/documents/${documentId}`,
            {convertResponseKeysToCamelCase: true},
        );
    }

    createPassengerDocument({
        uid,
        passengerId,
        document,
    }: INotebookCreatePassengerDocumentApiParams): Promise<TNotebookCreatePassengerDocumentApiResponse> {
        return this.post<TNotebookCreatePassengerDocumentApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/documents`,
            snakeCaseKeys(document as any, {deep: true}),
            {convertResponseKeysToCamelCase: true},
        );
    }

    editPassengerDocument({
        uid,
        passengerId,
        documentId,
        document,
    }: INotebookEditPassengerDocumentApiParams): Promise<TNotebookEditPassengerDocumentApiResponse> {
        return this.put<TNotebookEditPassengerDocumentApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/documents/${documentId}`,
            snakeCaseKeys(document as any, {deep: true}),
            {convertResponseKeysToCamelCase: true},
        );
    }

    deletePassengerDocument({
        uid,
        passengerId,
        documentId,
    }: INotebookDeletePassengerDocumentApiParams): Promise<TNotebookDeletePassengerDocumentApiResponse> {
        return this.delete<TNotebookDeletePassengerDocumentApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/documents/${documentId}`,
            {convertResponseKeysToCamelCase: true},
        );
    }

    getPassengerBonusCards({
        uid,
        passengerId,
    }: INotebookGetPassengerBonusCardsApiParams): Promise<TNotebookGetPassengerBonusCardsApiResponse> {
        return this.get<TNotebookGetPassengerBonusCardsApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/bonus-cards`,
            {convertResponseKeysToCamelCase: true},
        );
    }

    getPassengerBonusCard({
        uid,
        passengerId,
        bonusCardId,
    }: INotebookGetPassengerBonusCardApiParams): Promise<TNotebookGetPassengerBonusCardApiResponse> {
        return this.get<TNotebookGetPassengerBonusCardApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/bonus-cards/${bonusCardId}`,
            {convertResponseKeysToCamelCase: true},
        );
    }

    createPassengerBonusCard({
        uid,
        passengerId,
        bonusCard,
    }: INotebookCreatePassengerBonusCardApiParams): Promise<TNotebookCreatePassengerBonusCardApiResponse> {
        return this.post<TNotebookCreatePassengerBonusCardApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/bonus-cards`,
            snakeCaseKeys(bonusCard as any, {deep: true}),
            {convertResponseKeysToCamelCase: true},
        );
    }

    editPassengerBonusCard({
        uid,
        passengerId,
        bonusCardId,
        bonusCard,
    }: INotebookEditPassengerBonusCardApiParams): Promise<TNotebookEditPassengerBonusCardApiResponse> {
        return this.put<TNotebookEditPassengerBonusCardApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/bonus-cards/${bonusCardId}`,
            snakeCaseKeys(bonusCard as any, {deep: true}),
            {convertResponseKeysToCamelCase: true},
        );
    }

    deletePassengerBonusCard({
        uid,
        passengerId,
        bonusCardId,
    }: INotebookDeletePassengerBonusCardApiParams): Promise<TNotebookDeletePassengerBonusCardApiResponse> {
        return this.delete<TNotebookDeletePassengerBonusCardApiResponse>(
            `/travelers/${uid}/passengers/${passengerId}/bonus-cards/${bonusCardId}`,
            {convertResponseKeysToCamelCase: true},
        );
    }

    getDocumentTypes(): Promise<TNotebookGetDocumentTypesApiResponse> {
        return this.get<TNotebookGetDocumentTypesApiResponse>(
            `/document_types`,
        );
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            [CommonHeaders.X_YA_USER_TICKET]: this.userTicket,
        };
    }
}
