import {CommonHeaders} from 'server/constants/headers';

import {EApiEntry} from 'types/EApiEntry';
import {ESubscriptionAuthType} from 'types/subscription/ESubscriptionAuthType';
import {
    INotifierPromoConfig,
    INotifierSubscribeApiParams,
    INotifierSubscribeApiResponse,
    INotifierSubscriptionStatus,
} from 'server/api/NotifierApi/types';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from 'server/getContainerConfig';

export class NotifierApi extends RestApiClient {
    private readonly serviceTicket: string;

    constructor({
        logger,
        httpClient,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        requestId,
        rootSpan,
        getServiceTicket,
    }: IDependencies) {
        super({
            baseURL: `${getApiHost(EApiEntry.TRAVEL)}/subscriptions/v1`,
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
    }

    getStatus(
        email: string,
        authType: ESubscriptionAuthType,
        authValue: string,
    ): Promise<INotifierSubscriptionStatus> {
        return this.get<INotifierSubscriptionStatus>('get_status', {
            params: {
                authType,
                authValue,
                email,
            },
            timeout: 3000,
        });
    }

    getPromoConfig(params: {
        vertical: ESubscriptionVerticalName;
        authType: ESubscriptionAuthType;
        authValue: string;
        email?: string;
    }): Promise<INotifierPromoConfig> {
        return this.get<INotifierPromoConfig>('get_promo_config', {
            params,
        });
    }

    subscribe({
        authType,
        authValue,
        travelVerticalName,
        timezone,
        promoSubscriptionCode,
        nationalVersion,
        name = '',
        language,
        source,
        email,
    }: INotifierSubscribeApiParams): Promise<INotifierSubscribeApiResponse> {
        return this.post<INotifierSubscribeApiResponse>('subscribe', {
            credentials: [[authType, authValue]],
            travelVerticalName,
            timezone,
            promoSubscriptionCode,
            nationalVersion,
            name,
            language,
            source,
            email,
        });
    }

    unsubscribe(
        id: string,
        authType: ESubscriptionAuthType,
        authValue: string,
    ): Promise<void> {
        return this.get<void>('unsubscribe', {
            params: {
                authType,
                authValue,
                id,
            },
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }
}
