import {CommonHeaders} from 'server/constants/headers';

import {IOrdersGetOrderHappyPageApiParams} from 'server/api/OrdersAPI/types/IOrdersGetOrderHappyPageApiParams';
import {TOrdersGetOrderHappyPageApiResponse} from 'server/api/OrdersAPI/types/TOrdersGetOrderHappyPageApiResponse';
import {IOrdersIOrdersGetUserPromoAttributesApiResponse} from 'server/api/OrdersAPI/types/IOrdersIOrdersGetUserPromoAttributesApiResponse';
import {EApiEntry} from 'types/EApiEntry';
import {
    ICheckOrderAuthorizationResponse,
    TCheckOrderAuthorizationRequestParams,
} from 'server/api/OrdersAPI/types/ICheckOrderAuthorization';
import {
    ICreateOrderAuthorizationApiRequestBody,
    ICreateOrderAuthorizationResponse,
    TCreateOrderAuthorizationApiRequestQuery,
    TCreateOrderAuthorizationRequestParams,
} from 'server/api/OrdersAPI/types/ICreateOrderAuthorization';
import {TGetPaymentTestContextTokenParams} from 'server/api/OrdersAPI/types/TGetPaymentTestContextTokenParams';
import IGetPaymentTestContextTokenResponse from 'server/api/OrdersAPI/types/IGetPaymentTestContextTokenResponse';

import {RestApiClient} from 'server/utilities/RestApiClient';
import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';

import {IDependencies} from 'server/getContainerConfig';

export class OrdersAPI extends RestApiClient {
    private readonly serviceTicket: string;
    private readonly yandexuid: string;

    constructor({
        logger,
        requestId,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        httpClient,
        rootSpan,
        yandexuid,
    }: IDependencies) {
        super({
            baseURL: getApiHost(EApiEntry.TRAVEL),
            logger: logger,
            requestId,
            httpClient,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
        this.yandexuid = yandexuid;
    }

    /* Fetch orders flow */

    getOrderHappyPage(
        params: IOrdersGetOrderHappyPageApiParams,
    ): Promise<TOrdersGetOrderHappyPageApiResponse> {
        return this.get('/orders/v1/get_order_happy_page', {
            params,
            headers: {
                [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
            },
        });
    }

    getUserPromoAttributes(): Promise<IOrdersIOrdersGetUserPromoAttributesApiResponse> {
        return this.get('/promo_codes/v1/user_promo_attributes', {
            headers: {
                [CommonHeaders.X_YA_YANDEXUID]: this.yandexuid,
                [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            },
        });
    }

    /* Authorization order flow */

    checkOrderAuthorization(
        params: TCheckOrderAuthorizationRequestParams,
    ): Promise<ICheckOrderAuthorizationResponse> {
        return this.get<ICheckOrderAuthorizationResponse>(
            '/orders/v1/check_authorization',
            {
                params,
                headers: {
                    [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
                },
            },
        );
    }

    createOrderAuthorization(
        params: TCreateOrderAuthorizationRequestParams,
    ): Promise<ICreateOrderAuthorizationResponse> {
        const {secret, ...restParams} = params;

        const body: ICreateOrderAuthorizationApiRequestBody = {secret};
        const query: TCreateOrderAuthorizationApiRequestQuery = restParams;

        return this.post('/orders/v1/request_authorization', body, {
            params: query,
            headers: {
                [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
            },
        });
    }

    getPaymentTestContextToken(
        params: TGetPaymentTestContextTokenParams,
    ): Promise<IGetPaymentTestContextTokenResponse> {
        return this.get('/test_context/v1/payment_token', {
            params,
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.timeout = 40000;
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
        };
    }
}
